# coding=utf-8

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""Automatic tests for category model."""

from __future__ import unicode_literals

import unittest
from hamcrest import *

from PyQt5.Qt import Qt
from PyQt5 import Qt as Q

from asterstudy.common import external_files_callback, FilesSupplier
from asterstudy.datamodel.command import Unit
from asterstudy.datamodel import FileAttr
from asterstudy.gui.datafiles.model import Model, ModelItem
from asterstudy.gui.behavior import behavior
from asterstudy.gui import Entity, Role, NodeType
from asterstudy.gui.study import Study

import testutils.gui_utils

FileData_file = 0
FileData_unit = 1
FileData_inout = 2
FileData_exists = 3
FileData_embedded = 4

#------------------------------------------------------------------------------
def test_file_model():
    #--------------------------------------------------------------------------
    # Force default value of 'show catalogue name' option
    behavior().show_catalogue_name_data_files = True

    #--------------------------------------------------------------------------
    study = Study(None)
    history = study.history
    case = history.current_case

    #--------------------------------------------------------------------------
    model = study.dataFilesModel()
    assert_that(model, not_none())

    #--------------------------------------------------------------------------
    # check header data
    assert_that(model.headerData(FileData_inout, Qt.Horizontal), equal_to('Mode'))
    assert_that(model.headerData(FileData_unit, Qt.Horizontal), equal_to('Unit'))
    assert_that(model.headerData(FileData_file, Qt.Horizontal), equal_to('Filename'))
    assert_that(model.headerData(FileData_exists, Qt.Horizontal), equal_to('Exists'))
    assert_that(model.headerData(FileData_embedded, Qt.Horizontal), equal_to('Embedded'))
    assert_that(model.sourceModel().headerData(0, Qt.Vertical), equal_to(1))

    #--------------------------------------------------------------------------
    # check top-most root (invisible) item
    # - root item corresponds to invalid model index
    parent = Q.QModelIndex()
    # - model has 5 columns
    assert_that(model.columnCount(parent), equal_to(5))
    # - there is no rows yet (no stage)
    assert_that(model.rowCount(parent), equal_to(1))

    #--------------------------------------------------------------------------
    # create stage
    stage = case.create_stage(':memory:')
    model.update()

    #--------------------------------------------------------------------------
    # check root item
    # - root has 1 child item (stage) and 5 columns
    assert_that(model.rowCount(parent), equal_to(2))
    assert_that(model.columnCount(parent), equal_to(5))

    #--------------------------------------------------------------------------
    # check case item

    # case item: index (0, 0)
    index = model.index(0, FileData_file, parent)
    # - check parent
    assert_that(index.parent(), equal_to(parent))
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Case))
    # - check id
    data = model.data(index, Role.IdRole)
    assert_that(data, equal_to(case.uid))
    # - check user role: case
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(case))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to(case.name))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('<pre>Case: <b>{}</b></pre>'.format(case.name)))
    # - check icon (NOTE: pixmap is only available in app)
    data = model.data(index, Qt.DecorationRole)
    assert_that(data, none())
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, none())
    behavior().sort_stages = True
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to(0))
    behavior().sort_stages = False

    #--------------------------------------------------------------------------
    # check stage item

    # stage item: index (1, 0)
    index = model.index(1, FileData_file, parent)
    # - check parent
    assert_that(index.parent(), equal_to(parent))
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Stage))
    # - check id
    data = model.data(index, Role.IdRole)
    assert_that(data, equal_to(stage.uid))
    # - check user role: stage
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(stage))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to(stage.name))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('<pre>Stage: <b>{}</b></pre>'.format(stage.name)))
    # - check icon (NOTE: pixmap is only available in app)
    data = model.data(index, Qt.DecorationRole)
    assert_that(data, none())
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, none())
    behavior().sort_stages = True
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to(stage.number))
    behavior().sort_stages = False

    # stage item: index (1, 1)
    index = model.index(1, FileData_unit, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Stage))
    # - check id
    data = model.data(index, Role.IdRole)
    assert_that(data, equal_to(stage.uid))
    # - check user role: stage
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(stage))
    # - check title (should be None)
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, none())
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, none())
    behavior().sort_stages = True
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to(stage.number))
    behavior().sort_stages = False

    #--------------------------------------------------------------------------
    # add command
    command = stage('PRE_GMSH')

    # in this command:
    #    - UNITE_GMSH is keyword of type 'in'
    #    - UNITE_MAILLAGE is keyword of type 'out'

    #--------------------------------------------------------------------------
    # initialize a keyword that refers to 'in' file
    unit = command['UNITE_GMSH']
    unit.value = {19: 'gmsh.file'}
    assert_that(unit, instance_of(Unit))
    model.update()

    #--------------------------------------------------------------------------
    # check file item

    # set stage as parent
    parent = model.index(1, FileData_file, Q.QModelIndex())
    # - stage has 1 child item (file) and 5 columns
    assert_that(model.rowCount(parent), equal_to(1))
    assert_that(model.columnCount(parent), equal_to(5))

    # file item: index (0, 0)
    index = model.index(0, FileData_file, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Unit))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to(unit.filename))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('File: ' + unit.filename))
    # - check icon (NOTE: pixmap is only available in app)
    data = model.data(index, Qt.DecorationRole)
    assert_that(data, none())
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to(unit.filename))
    # - check background role

    data = model.data(index, Qt.BackgroundRole)
    assert_that(data, equal_to(Q.QBrush(Q.QColor(Q.Qt.white))))
    # - check user role: filename
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to('gmsh.file'))

    # file item: index (0, 1)
    index = model.index(0, FileData_unit, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Unit))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to(unit.value))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to(unit.value))
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to(unit.value))
    # - check user role: unit
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(unit.value))

    # file item: index (0, 2)
    index = model.index(0, FileData_inout, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Unit))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('in'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('in'))
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to('in'))
    # - check user role: inout attribute
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(FileAttr.In))

    # file item: index (0, 3)
    index = model.index(0, FileData_exists, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Unit))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('No'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('No'))
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to('No'))
    # - check user role: exists attribute
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(False))

    # file item: index (0, 4)
    index = model.index(0, FileData_embedded, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Unit))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('No'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('No'))
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to('No'))
    # - check user role: embedded attribute
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(False))

    #--------------------------------------------------------------------------
    # switch ON embedded state
    info = stage.handle2info[19]
    info.embedded = True
    model.update()

    #--------------------------------------------------------------------------
    # check file item

    # re-get stage item
    parent = model.index(1, FileData_file, Q.QModelIndex())

    # file item: index (0, 0)
    index = model.index(0, FileData_file, parent)
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to(unit.filename + ' (embedded)'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('File: ' + unit.filename + ' (embedded)'))
    # - check user role: filename
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to('gmsh.file'))

    # file item: index (0, 4)
    index = model.index(0, FileData_embedded, parent)
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('Yes'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('Yes'))
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to('Yes'))
    # - check user role: Embedded attribute
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(True))

    #--------------------------------------------------------------------------
    # check command item

    # set file as parent
    parent = index
    # - file has 1 child item (command) and 5 columns
    assert_that(model.rowCount(index), equal_to(1))
    assert_that(model.columnCount(index), equal_to(5))

    # command item: index (0, 0)
    index = model.index(0, FileData_file, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Command))
    # - check id
    data = model.data(index, Role.IdRole)
    assert_that(data, equal_to(command.uid))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('[noname] (PRE_GMSH)'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('<pre>Command: <b>[noname]</b> (PRE_GMSH)<br>From stage: <b>:memory:</b></pre>'))
    # - check icon (NOTE: pixmap is only available in app)
    data = model.data(index, Qt.DecorationRole)
    assert_that(data, none())
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to('[noname]'))
    # - check user role: command
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(command))

    # command item: index (0, 1)
    index = model.index(0, FileData_unit, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Command))
    # - check id
    data = model.data(index, Role.IdRole)
    assert_that(data, equal_to(command.uid))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to(command.uid))
    # - check user role: command
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(command))

    #--------------------------------------------------------------------------
    # initialize a keyword that refers to 'out' file
    unit = command['UNITE_MAILLAGE']
    unit.value = {20: 'maillage.file'}
    assert_that(unit, instance_of(Unit))
    model.update()

    #--------------------------------------------------------------------------
    # check file item

    # set stage as parent
    parent = model.index(1, FileData_file, Q.QModelIndex())
    # - stage has 2 child items (file) and 5 columns
    assert_that(model.rowCount(parent), equal_to(2))
    assert_that(model.columnCount(parent), equal_to(5))

    # file item: index (1, 0)
    index = model.index(1, FileData_file, parent)
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to(unit.filename))

    # file item: index (1, 2)
    index = model.index(1, FileData_inout, parent)
    assert_that(index.parent(), equal_to(parent))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('out'))
    # check user role: InOut attribute
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(FileAttr.Out))

    #--------------------------------------------------------------------------
    # modify a keyword that refers to 'in' file: older unit 19 no longer used
    unit = command['UNITE_GMSH']
    unit.value = {20: 'folder/maillage.file'}
    model.update()

    #--------------------------------------------------------------------------
    # check file item

    # set stage as parent
    parent = model.index(1, FileData_file, Q.QModelIndex())
    # - stage has 2 child items (file)
    assert_that(model.rowCount(parent), equal_to(2))

    # stage item: index (0, 2)
    index = model.index(0, FileData_inout, parent)
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('in'))
    # - check user role: InOut attribute
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(FileAttr.In))

    # stage item: index (1, 2)
    index = model.index(1, FileData_inout, parent)
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('inout'))
    # - check user role: InOut attribute
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(FileAttr.InOut))

    #--------------------------------------------------------------------------
    # modify a keyword that refers to 'out' file

    unit = command['UNITE_MAILLAGE']

    # 1. change unit value with undefined file name
    # while not associated to a command,
    # older value 'maillage.file' is kept in memory (stage)
    unit.value = 21
    model.update()

    # check file item

    # set stage as parent
    parent = model.index(1, FileData_file, Q.QModelIndex())
    # - stage has 3 child items (file)
    assert_that(model.rowCount(parent), equal_to(3))

    # stage item: index (2, 0)
    index = model.index(2, FileData_file, parent)
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('<undefined>'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('File: <undefined>'))

    # 2. change filename
    stage.handle2info[unit.value].filename = 'dir/maillage.file'

    # check file item

    # set stage as parent
    parent = model.index(1, FileData_file, Q.QModelIndex())
    # - stage has 3 child items (file)
    assert_that(model.rowCount(parent), equal_to(3))

    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('maillage.file'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('File: dir/maillage.file'))

    #--------------------------------------------------------------------------
    # check that stage is in graphical mode
    assert_that(stage.is_graphical_mode(), equal_to(True))

    index = model.index(0, FileData_inout, parent)
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(FileAttr.In))
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('in'))

    #--------------------------------------------------------------------------
    # switch stage to text mode
    stage.use_text_mode()
    model.update()
    #
    # at this point, the text stage reads:
    #
    # PRE_GMSH(UNITE_GMSH=21,
    #          UNITE_MAILLAGE=20)

    assert_that(stage.is_text_mode(), equal_to(True))

    parent = model.index(1, FileData_file, Q.QModelIndex())

    # since 'maillage.file' is no longer used,
    # it has been removed from memory
    assert_that(model.rowCount(parent), equal_to(2))

    index = model.index(0, FileData_inout, parent)
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(FileAttr.In))
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('in'))

    stage.handle2info[unit.value].attr = FileAttr.Out
    model.update()

    parent = model.index(1, FileData_file, Q.QModelIndex())

    index = model.index(1, FileData_inout, parent)
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(FileAttr.Out))
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('out'))

    #--------------------------------------------------------------------------
    # switch stage to graphical mode
    stage.use_graphical_mode()
    model.update()

    assert_that(stage.is_graphical_mode(), equal_to(True))

    parent = model.index(1, FileData_file, Q.QModelIndex())

    # 'inout' value reinitialized according to catalog when converting back to graphical
    # whatever the one provided by the user in text mode
    index = model.index(0, FileData_inout, parent)
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(FileAttr.In))
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('in'))

    #--------------------------------------------------------------------------
    # check that 'show catalogue name' option is taken into account
    behavior().show_catalogue_name_data_files = False
    model.update()

    parent = model.index(1, FileData_file, Q.QModelIndex()) # stage
    parent = model.index(1, FileData_file, parent)          # file
    index = model.index(0, FileData_file, parent)           # command
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('[noname]'))

    #--------------------------------------------------------------------------
    pass


def test_reference():
    #--------------------------------------------------------------------------
    class Supplier(FilesSupplier):
        _files = {'0:1:1:1':"referenced object"}
        def files(self, file_format):
            return self._files.keys()
        def filename(self, uid):
            return self._files.get(uid)

    supplier = Supplier()
    external_files_callback(supplier, True)

    study = Study(None)
    history = study.history
    case = history.current_case
    stage = case.create_stage(':memory:')
    command = stage('LIRE_MAILLAGE')
    command.init({'UNITE':{20: '0:1:1:1'}})

    #--------------------------------------------------------------------------
    model = study.dataFilesModel()
    model.update()
    assert_that(model, not_none())

    #--------------------------------------------------------------------------
    parent = model.index(1, FileData_file, Q.QModelIndex())
    assert_that(model.rowCount(parent), equal_to(1))

    #--------------------------------------------------------------------------

    # file item: index (0, 0) - filename
    index = model.index(0, FileData_file, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Unit))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('referenced object'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('File: referenced object (0:1:1:1)'))
    # - check icon (NOTE: pixmap is only available in app)
    data = model.data(index, Qt.DecorationRole)
    assert_that(data, none())
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to('referenced object'))
    # - check user role: filename
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to('0:1:1:1'))
    # - check reference role
    data = model.data(index, Role.ReferenceRole)
    assert_that(data, equal_to(True))

    # file item: index (0, 1) - unit
    index = model.index(0, FileData_unit, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Unit))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to(20))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to(20))
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to(20))
    # - check user role: filename
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(20))
    # - check reference role
    data = model.data(index, Role.ReferenceRole)
    assert_that(data, equal_to(True))

    # file item: index (0, 2) - inout attribute
    index = model.index(0, FileData_inout, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Unit))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('in'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('in'))
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to('in'))
    # - check user role: inout attribute
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(FileAttr.In))
    # - check reference role
    data = model.data(index, Role.ReferenceRole)
    assert_that(data, equal_to(True))

    # file item: index (0, 3) - exists attribute
    index = model.index(0, FileData_exists, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Unit))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('Yes'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('Yes'))
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to('Yes'))
    # - check user role: exists attribute
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(True))
    # - check reference role
    data = model.data(index, Role.ReferenceRole)
    assert_that(data, equal_to(True))

    # file item: index (0, 4) - embedded attribute
    index = model.index(0, FileData_embedded, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Unit))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(True))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    # A SMESH file desc should not be considered embedded
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('No'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('No'))
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to('No'))
    # - check user role: Embedded attribute
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to(False))
    # - check reference role
    data = model.data(index, Role.ReferenceRole)
    assert_that(data, equal_to(True))

    #--------------------------------------------------------------------------
    external_files_callback(supplier, False)
    model.update()

    #--------------------------------------------------------------------------
    parent = model.index(1, FileData_file, Q.QModelIndex())
    assert_that(model.rowCount(parent), equal_to(1))

    #--------------------------------------------------------------------------
    index = model.index(0, FileData_file, parent)
    # - check type
    data = model.data(index, Role.TypeRole)
    assert_that(data, equal_to(NodeType.Unit))
    # - check validity
    data = model.data(index, Role.ValidityRole)
    assert_that(data, equal_to(False))
    # - check visibility
    data = model.data(index, Role.VisibilityRole)
    assert_that(data, equal_to(True))
    # - check title
    data = model.data(index, Qt.DisplayRole)
    assert_that(data, equal_to('<undefined>'))
    # - check tooltip
    data = model.data(index, Qt.ToolTipRole)
    assert_that(data, equal_to('File: <undefined> (0:1:1:1)'))
    # - check icon (NOTE: pixmap is only available in app)
    data = model.data(index, Qt.DecorationRole)
    assert_that(data, none())
    # - check sort role
    data = model.data(index, Role.SortRole)
    assert_that(data, equal_to('<undefined>'))
    # - check user role: filename
    data = model.data(index, Qt.UserRole)
    assert_that(data, equal_to('0:1:1:1'))
    # - check reference role
    data = model.data(index, Role.ReferenceRole)
    assert_that(data, equal_to(True))

    #--------------------------------------------------------------------------
    pass

def test_highlight_files():
    """Test files that appear twice are highlighted"""

    study = Study(None)
    history = study.history
    case = history.current_case

    #--------------------------------------------------------------------------
    model = study.dataFilesModel()
    assert_that(model, not_none())

    st1 = case.create_stage(':st1:')
    st2 = case.create_stage(':st2:')

    cmd = st1('LIRE_MAILLAGE')
    cmd.init({'UNITE':{20: '/my/duplicate/name'}})

    cmd = st2('LIRE_MAILLAGE')
    cmd.init({'UNITE':{20: '/my/duplicate/name'}})

    model.update()
    assert_that(model, not_none())

    # first stage item: index (2, 0)
    # cause there are two *DirItem* objects first
    parent = model.index(2, FileData_file, Q.QModelIndex())

    # file item within stage item: index (0, 0)
    index = model.index(0, FileData_file, parent)

    data = model.data(index, Qt.BackgroundRole)
    assert_that(data, equal_to(Q.QBrush(Q.QColor(Q.Qt.yellow))))



if __name__ == "__main__":
    import sys
    from testutils import get_test_suite
    RET = unittest.TextTestRunner(verbosity=2).run(get_test_suite(__name__))
    sys.exit(not RET.wasSuccessful())
