# -*- coding: utf-8 -*-

# Copyright 2016 - 2018 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""Automatic tests for parametric studies."""

from __future__ import print_function, unicode_literals

import os
import os.path as osp
import unittest

from asterstudy.api import ParametricCalculation
from asterstudy.common import debug_message
from asterstudy.datamodel.comm2study import comm2study
from asterstudy.datamodel.history import History
from asterstudy.datamodel.parametric import export_to_openturns, output_commands
from asterstudy.datamodel.study2comm import study2comm
from asterstudy.datamodel.engine.engine_utils import ExportCase
from asterstudy.datamodel.engine.salome_runner import has_salome
from asterstudy.datamodel.engine import runner_factory, Engine
from asterstudy.datamodel.result import StateOptions as SO
from engine_testcases import monitor_refresh, _parameters
from hamcrest import *
from testutils import tempdir

_multiprocess_can_split_ = False


def _setup_nominal(tmpdir):
    history = History()
    if not history.support_parametric:
        return None

    history.folder = tmpdir
    case = history.current_case
    case.name = 'beam'

    # first stage
    stage1 = case.create_stage('deflection')
    cmd_file = osp.join(os.getenv('ASTERSTUDYDIR'),
                        'data', 'export', 'parametric_beam.comm')
    with open(cmd_file, 'rb') as comm:
        comm2study(comm.read(), stage1)

    info = stage1.handle2info[20]
    info.filename = osp.join(os.getenv('ASTERSTUDYDIR'),
                             'data', 'export', 'sslv159b.mail')

    # second stage containing IMPR_TABLE
    stage2 = case.create_stage('output')
    cmd_file = osp.join(os.getenv('ASTERSTUDYDIR'),
                        'data', 'export', 'parametric_beam.com1')
    with open(cmd_file, 'rb') as comm:
        comm2study(comm.read(), stage2)

    info = stage2.handle2info[10]
    info.filename = '/path/to/user/result_file.npy'

    # third stage containing unnecessary commands
    stage3 = case.create_stage('ignored')
    cmd_file = osp.join(os.getenv('ASTERSTUDYDIR'),
                        'data', 'export', 'parametric_beam.com2')
    with open(cmd_file, 'rb') as comm:
        comm2study(comm.read(), stage3)

    # not called by open_openturns!
    param_case = history.create_parametric_case()
    return param_case

def _setup_and_check_ot(case, tmpdir, mem_limit=None):
    # inputs for the widget
    varnames = case.variables.keys()
    outcmd = output_commands(case)

    # outputs of the widget: user's selection : 'input_vars' & 'command'
    input_vars = ["E", "F"]
    assert_that(varnames, has_items(*input_vars))
    assert_that(outcmd, has_length(1))
    command = outcmd[0]

    code = export_to_openturns(case, input_vars, command)
    debug_message(("-" * 30 + "\n{0}\n" + "-" * 30).format(code))
    assert_that(code.strip().startswith("def _exec("))
    assert_that(code,
                matches_regexp(r"ParametricCalculation\([\'\"]{0}[\'\"]"
                               .format(case.folder)))

    export = osp.join(tmpdir, "ParamCase_1", "parametric.export")
    with open(export, 'rb') as fexp:
        content = fexp.read()
    assert_that(content, contains_string("parametric.comm"))
    assert_that(content, contains_string("parametric.com1"))
    assert_that(content, is_not(contains_string("parametric.com2")))
    # check that parameters are taken from the previous execution
    assert_that(content, contains_string("memory_limit %s" % (mem_limit or 2048)))

    # next steps are done in OpenTurns
    # check with F = twice the nominal value
    context = {}
    # run silently in unittest
    code = code.replace("set_verbosity(1)", "set_verbosity(0)")
    exec code in context
    results = context['_exec'](2.e11, -1000.)

    assert_that(results, has_length(3))
    assert_that(results[0], equal_to(0.5))
    assert_that(results[1], equal_to(0.))
    assert_that(round(results[2], 6), equal_to(-0.005373))

    # check that files are deleted in case of success and not in debug mode
    # it will fail in debug mode: useful to keep temporary files
    assert_that(osp.exists(osp.join(tmpdir, "ParamCase_1", ".run_param0001")),
                equal_to(False))


def test_parametric_study2comm():
    """Test generator of parametric command file"""
    case = _setup_nominal("unset")
    if case is None:
        return

    varnames = case.variables.keys()
    assert_that(varnames, contains("E", "nu", "F", "h"))

    stage = case[0]
    text = study2comm(stage, parametric=True)
    assert_that(text, contains_string("E = VARIABLE(NOM_PARA="))
    assert_that(text, contains_string("nu = VARIABLE(NOM_PARA="))
    assert_that(text, contains_string("F = VARIABLE(NOM_PARA="))
    assert_that(text, contains_string("h = VARIABLE(NOM_PARA="))

@tempdir
def test_export_parametric(tmpdir):
    case = _setup_nominal(tmpdir)
    if case is None:
        return

    export_name = osp.join(tmpdir, "parametric.export")
    export = ExportCase.factory(case, export_name, parametric=True)
    params = {'server': 'remote', 'memory': 4096, 'time': '1:00:00',
              'mpicpu': 'x'}
    export.set_parameters(params)
    export.generate()

    with open(export_name, 'rb') as fexp:
        content = fexp.read()

    assert_that(content, contains_string("server remote"))
    assert_that(content, contains_string("memory_limit 4096"))
    assert_that(content, contains_string("time_limit 3600"))

@unittest.skipIf(not has_salome(), "salome is required")
@tempdir
def test_parametric_generator(tmpdir):
    """Test execution of parametric study"""
    case = _setup_nominal(tmpdir)
    if case is None:
        return

    assert_that(case, has_length(3))

    # add a previous execution
    engine = Engine.Salome
    rc1 = case.model.create_run_case(name='fake')
    result = rc1.results()[2]
    runner = runner_factory(engine, case=rc1, unittest=True)
    params = _parameters(engine)
    params['memory'] = 1234
    runner.start(params)
    monitor_refresh(runner, estimated_time=5.)
    assert_that(runner.result_state(result) & SO.Success)

    _setup_and_check_ot(case, tmpdir, mem_limit=1234)

@unittest.skipIf(not has_salome(), "salome is required")
@tempdir
def test_parametric_generator_wo_run(tmpdir):
    """Test execution of parametric study"""
    case = _setup_nominal(tmpdir)
    if case is None:
        return

    assert_that(case, has_length(3))

    _setup_and_check_ot(case, tmpdir)

@tempdir
def test_parametric_coverage(tmpdir):
    # for coverage
    calc = ParametricCalculation(tmpdir, ["x0"], ["y0", "y1"])
    calc._rcname = None
    name = calc.run_case_name()
    assert_that(osp.exists(osp.join(tmpdir, name + '.mark')))
    calc.runcdir = osp.join(calc.basedir, calc.run_case_name())

    calc.set_verbosity(0)
    res = calc.read_output_file()
    assert_that(res, none())
    calc.set_verbosity(1)
    calc._print("")


if __name__ == "__main__":
    import sys
    from testutils import get_test_suite
    RET = unittest.TextTestRunner(verbosity=2).run(get_test_suite(__name__))
    sys.exit(not RET.wasSuccessful())
