# -*- coding: utf-8 -*-

# Copyright 2017 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""Automatic tests for running services."""

from __future__ import unicode_literals

import os
import os.path as osp
import subprocess as SP
import tempfile
import unittest

from asterstudy.datamodel.engine import Engine, version_ismpi
from asterstudy.datamodel.engine.engine_utils import (asrun_diag,
                                                      convert_asrun_state,
                                                      convert_launcher_state,
                                                      create_profil_for_current,
                                                      export_comm, kill_aster)
from asterstudy.datamodel.general import FileAttr
from asterstudy.datamodel.history import History
from asterstudy.datamodel.result import StateOptions as SO
from engine_testcases import DEFAULT_PARAMS, _setup_history
from hamcrest import *
from testutils import tempdir


def test_diagnostic_asrun():
    """Test for diagnostic conversion for AsRun"""
    states = ('_', 'PEND', 'SUSPENDED', 'RUN', 'ENDED')
    diags = (
        'OK', 'NOOK_TEST_RESU', 'NO_TEST_RESU', '<A>_ALARM',
        '<S>_ERROR', '<S>_CPU_LIMIT', '<S>_MEMORY_ERROR', '<S>_NO_CONVERGENCE',
        '<S>_CONTACT_ERROR',
        '<F>_ERROR', '<F>_SYNTAX_ERROR', '<F>_ABNORMAL_ABORT',
        '<F>_SUPERVISOR',
    )
    # loop on all possibilities
    for state in states:
        for diag in diags:
            ret = convert_asrun_state(state, diag)
            if state == 'RUN':
                assert_that(ret & SO.Finished, equal_to(0))
                assert_that(ret & SO.NotFinished)
                assert_that(ret & SO.Running)
                assert_that(ret & SO.Warn, equal_to(0))
                assert_that(ret & SO.CpuLimit, equal_to(0))
            elif state != 'ENDED':
                assert_that(ret & SO.Finished, equal_to(0))
                assert_that(ret & SO.NotFinished)
                assert_that(ret & SO.Pending)
                assert_that(ret & SO.Warn, equal_to(0))
                assert_that(ret & SO.CpuLimit, equal_to(0))
            else:
                assert_that(ret & SO.Finished)
                assert_that(ret & SO.NotFinished, equal_to(0))
                if '<A>' in diag:
                    assert_that(ret & SO.Warn)
                if '<S>' in diag:
                    assert_that(ret & SO.Interrupted)
                if '<F>' in diag:
                    assert_that(ret & SO.Error)
                if diag in ('NOOK_TEST_RESU', 'NO_TEST_RESU'):
                    assert_that(ret & SO.Nook)
    # test common combinaisons
    ret = convert_asrun_state('ENDED', '<A>_ALARM')
    assert_that(ret & SO.Finished)
    assert_that(ret & SO.Success)
    assert_that(ret & SO.Warn)

    ret = convert_asrun_state('ENDED', '<S>_CPU_LIMIT')
    assert_that(ret & SO.Finished)
    assert_that(ret & SO.Interrupted)
    assert_that(ret & SO.CpuLimit)

    ret = convert_asrun_state('ENDED', '<F>_ERROR')
    assert_that(ret & SO.Finished)
    assert_that(ret & SO.Error)
    assert_that(ret & SO.Warn, equal_to(0))

    ret = SO.Intermediate
    assert_that(ret & SO.Finished)
    assert_that(ret & SO.Error, equal_to(0))


def test_diagnostic_salome():
    """Test for diagnostic conversion for Salome"""
    states = ('CREATED', 'IN_PROCESS', 'QUEUED', 'RUNNING', 'PAUSED',
              'FINISHED', 'FAILED')
    diags = (
        'OK', 'NOOK_TEST_RESU', 'NO_TEST_RESU', '<A>_ALARM',
        '<S>_ERROR', '<S>_CPU_LIMIT', '<S>_MEMORY_ERROR', '<S>_NO_CONVERGENCE',
        '<S>_CONTACT_ERROR',
        '<F>_ERROR', '<F>_SYNTAX_ERROR', '<F>_ABNORMAL_ABORT',
        '<F>_SUPERVISOR',
    )
    # loop on all possibilities
    for state in states:
        for diag in diags:
            ret = convert_launcher_state(state, diag)
            if state == 'RUNNING':
                assert_that(ret & SO.Finished, equal_to(0))
                assert_that(ret & SO.NotFinished)
                assert_that(ret & SO.Running)
                assert_that(ret & SO.Warn, equal_to(0))
                assert_that(ret & SO.CpuLimit, equal_to(0))
            elif state not in ('FINISHED', 'FAILED'):
                assert_that(ret & SO.Finished, equal_to(0))
                assert_that(ret & SO.NotFinished)
                assert_that(ret & SO.Pending)
                assert_that(ret & SO.Warn, equal_to(0))
                assert_that(ret & SO.CpuLimit, equal_to(0))
            else:
                assert_that(ret & SO.Finished)
                assert_that(ret & SO.NotFinished, equal_to(0))
                if '<A>' in diag:
                    assert_that(ret & SO.Warn)
                if '<S>' in diag:
                    assert_that(ret & SO.Interrupted)
                if '<F>' in diag:
                    assert_that(ret & SO.Error)
                if diag in ('NOOK_TEST_RESU', 'NO_TEST_RESU'):
                    assert_that(ret & SO.Nook)
    # test common combinaisons
    ret = convert_launcher_state('FINISHED', '<A>_ALARM')
    assert_that(ret & SO.Finished)
    assert_that(ret & SO.Success)
    assert_that(ret & SO.Warn)

    ret = convert_launcher_state('FINISHED', '<S>_CPU_LIMIT')
    assert_that(ret & SO.Finished)
    assert_that(ret & SO.Interrupted)
    assert_that(ret & SO.CpuLimit)

    ret = convert_launcher_state('FINISHED', '<F>_ERROR')
    assert_that(ret & SO.Finished)
    assert_that(ret & SO.Error)
    assert_that(ret & SO.Warn, equal_to(0))

    ret = convert_launcher_state('FAILED')
    assert_that(ret & SO.Finished)
    assert_that(ret & SO.Error)
    assert_that(ret & SO.Warn, equal_to(0))

    # check cputime limit error
    output = "CPU time limit exceeded"
    filename = tempfile.mkstemp(prefix='diag')[1]
    with open(filename, "wb") as fobj:
        fobj.write(output)
    diag = asrun_diag(filename)
    assert_that(diag, equal_to('<F>_CPU_LIMIT'))
    os.remove(filename)


def test_state_name():
    """Test for StateOptions names"""
    main = {
        SO.Waiting: "Waiting",
        SO.Pending: "Pending",
        SO.Running: "Running",
        SO.Pausing: "Pausing",
        SO.Success: "Success",
        SO.Error: "Error",
        SO.Interrupted: "Interrupted",
        SO.Intermediate: "Intermediate",
    }
    errors = {
        SO.Nook: "Nook",
        SO.CpuLimit: "CpuLimit",
        SO.NoConvergence: "NoConvergence",
        SO.Memory: "Memory",
    }
    for state, name in main.items():
        assert_that(SO.name(state), equal_to(name))
        state = state | SO.Intermediate
        if name != "Intermediate":
            assert_that(SO.name(state), equal_to(name + "+Intermediate"))
        else:
            assert_that(SO.name(state), equal_to(name))

    state = SO.Success | SO.Warn
    assert_that(SO.name(state), equal_to("Success+Warn"))

    state = SO.Error
    for add, name in errors.items():
        assert_that(SO.name(state | add), equal_to("Error+" + name))
        add = add | SO.Intermediate
        assert_that(SO.name(state | add),
                    equal_to("Error+" + name + "+Intermediate"))


@tempdir
def test_export_with_intermediate(tmpdir):
    """Test for creation of export with intermediate stages"""
    history = _setup_history(tmpdir)
    case = history.current_case
    params = DEFAULT_PARAMS.copy()
    # add a data file in stage s3
    info = case[2].handle2info[33]
    info.filename = osp.join(os.getenv('ASTERSTUDYDIR'),
                             'data', 'export', 'forma13a.23')
    info.attr = FileAttr.In

    stages = [case[i] for i in range(4)]
    prof = create_profil_for_current(None, case, stages,
                                     'test_inter', params, None)

    data = [osp.basename(i.path) for i in prof.get_data()]
    result = [osp.basename(i.path) for i in prof.get_result()]
    for i in range(4):
        if i == 0:
            i = 'm'
        assert_that(data, has_item('test_inter.com{0}'.format(i)))
    assert_that(data, has_item('fileB.38'))
    assert_that(data, has_item('forma13a.23'))
    assert_that(data, has_length(6))
    assert_that(result, has_item('fileA.44'))
    assert_that(result, has_item('fileB.38'))
    assert_that(result, has_item('fileC.39'))
    assert_that(result, has_item('message'))
    assert_that(result, has_item('base-stage4'))
    assert_that(result, has_length(5))


@tempdir
def test_comm_files(tmpdir):
    """Test for creation of comm files"""
    def _check_content(filename, starter, nbcmds):
        content = open(filename, 'rb').read()
        assert_that(content, contains_string(starter))
        assert_that(content, contains_string('FIN'))
        assert_that(content.count('='), equal_to(nbcmds))

    text1 = """mesh = MODI_MAILLAGE()"""
    text2 = """model = AFFE_MODELE()"""
    text3 = """resu = MECA_STATIQUE()"""

    history = History()
    history.folder = tmpdir
    case = history.current_case
    case.name = 'c1'
    s1 = case.create_stage('s1')
    s1.use_text_mode()
    s2 = case.create_stage('s2')
    s2.use_text_mode()
    s3 = case.create_stage('s3')
    s3.use_text_mode()
    stages = [s1, s2, s3]

    # should create one commands file
    s1.set_text(text1)
    s2.set_text(text2)
    s3.set_text(text3)
    files = export_comm(stages, 'test1')
    assert_that(files, has_length(1))
    _check_content(files[0], 'DEBUT', 3)

    # force separated stages
    files = export_comm(stages, 'test1bis', separated=True)
    assert_that(files, has_length(3))
    _check_content(files[0], 'DEBUT', 1)
    _check_content(files[1], 'POURSUITE', 1)
    _check_content(files[2], 'POURSUITE', 1)

    # first separated
    s1.set_text(os.linesep.join(["DEBUT()", text1, "FIN()"]))
    files = export_comm(stages, 'test2')
    assert_that(files, has_length(2))
    _check_content(files[0], 'DEBUT', 1)
    _check_content(files[1], 'POURSUITE', 2)

    # all separated because of the second
    s1.set_text(text1)
    s2.set_text(os.linesep.join(["POURSUITE()", text2, "FIN()"]))
    files = export_comm(stages, 'test3')
    assert_that(files, has_length(3))
    _check_content(files[0], 'DEBUT', 1)
    _check_content(files[1], 'POURSUITE', 1)
    _check_content(files[2], 'POURSUITE', 1)

    # last separated
    s2.set_text(text2)
    s3.set_text(os.linesep.join(["POURSUITE()", text3, "FIN()"]))
    files = export_comm(stages, 'test4')
    assert_that(files, has_length(2))
    _check_content(files[0], 'DEBUT', 2)
    _check_content(files[1], 'POURSUITE', 1)


@tempdir
def test_comm_files_error(tmpdir):
    """Test for creation of comm files"""
    history = History()
    history.folder = tmpdir
    case = history.current_case
    case.name = 'c1'
    s1 = case.create_stage('s1')
    s1.use_text_mode()
    invalid = "invalid Python syntax + 1."
    s1.set_text(invalid)
    files = export_comm([s1], 'job')
    assert_that(files, has_length(1))

    # text is unchanged
    with open(files[0], 'rb') as comm:
        assert_that(comm.read(), equal_to(invalid))


def test_tail():
    """Test for tail utility"""
    from asterstudy.datamodel.engine.engine_utils import remote_tail
    text = remote_tail("", "localhost", __file__, 2)
    assert_that(text, contains_string("TextTestRunner"))
    assert_that(text, is_not(contains_string("__main__")))


def test_unavailable_joblist():
    from asterstudy.datamodel.engine import NAJobsList
    from asterstudy.datamodel.engine.salome_runner import SalomeJobsList
    # does nothing
    assert_that(NAJobsList.load_jobs(""), none())
    assert_that(NAJobsList.save_jobs(), equal_to(""))

    # force error just for coverage
    SalomeJobsList._hdlr = object()
    assert_that(SalomeJobsList.load_jobs(""), none())
    assert_that(SalomeJobsList.load_jobs("<xml/>"), none())
    assert_that(SalomeJobsList.save_jobs(), equal_to(""))
    SalomeJobsList._hdlr = None


def test_mpi_version():
    assert_that(version_ismpi('stable'), equal_to(False))
    assert_that(version_ismpi('stable_mpi'), equal_to(True))
    assert_that(version_ismpi('14.0'), equal_to(False))
    assert_that(version_ismpi('14.0_mpi'), equal_to(True))


def test_kill():
    fake_aster = ['python', '-c', 'import time; time.sleep(10)',
                  '--num_job=MYID', '--mode=interactif']
    aster = SP.Popen(fake_aster)
    pid = aster.pid
    killed = kill_aster("MYID")
    aster.wait()
    assert_that(killed, equal_to(pid))

    aster = SP.Popen(fake_aster)
    killed = kill_aster("MYID", "", "localhost")
    aster.wait()
    assert_that(killed, equal_to(999999))


def test_default():
    from asterstudy.datamodel.engine import init_default_engine
    assert_that(Engine.Default, equal_to(Engine.Salome))
    init_default_engine(Engine.Salome)
    assert_that(Engine.Default, equal_to(Engine.Salome))


if __name__ == "__main__":
    import sys
    from testutils import get_test_suite
    RET = unittest.TextTestRunner(verbosity=2).run(get_test_suite(__name__))
    sys.exit(not RET.wasSuccessful())
