# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
ASTERSTUDY module
-----------------

Implementation of SALOME ASTERSTUDY module.

"""

from __future__ import unicode_literals

# pragma pylint: disable=import-error,no-name-in-module
from asterstudy.common import debug_message
from asterstudy.gui.salomegui import AsterSalomeGui, get_aster_view_type


def get_aster_gui():
    """
    Get ASTERSTUDY GUI instance.

    Returns:
        AsterSalomeGui: SALOME AsterStudy GUI object
    """
    if not hasattr(get_aster_gui, "aster_gui"):
        get_aster_gui.aster_gui = AsterSalomeGui()
    return get_aster_gui.aster_gui


# note: the following pragma is added to prevent pylint complaining
#       about functions that follow Qt naming conventions; below API
#       comes from SALOME
# pragma pylint: disable=invalid-name


def initialize():
    """Initialize module."""
    debug_message("ASTERSTUDYGUI::initialize()")
    get_aster_gui().initialize()


def activate():
    """Activate module."""
    debug_message("ASTERSTUDYGUI::activate()")

    import salome
    salome.salome_init()
    smesh = salome.myStudy.FindComponent(str('SMESH'))
    if smesh is not None:
        engine = salome.lcc.FindOrLoadComponent(str("FactoryServer"),
                                                str("SMESH"))
        builder = salome.myStudy.NewBuilder()
        builder.LoadWith(smesh, engine)

    return get_aster_gui().activate()


def deactivate():
    """Deactivate module."""
    debug_message("ASTERSTUDYGUI::deactivate()")
    get_aster_gui().deactivate()


def createPreferences():
    """Export preferences to common Preferences dialog."""
    debug_message("ASTERSTUDYGUI::createPreferences()")
    get_aster_gui().createPreferences()


def preferenceChanged(section, name):
    """Called when preferences item has been changed in Preferences dialog."""
    debug_message("ASTERSTUDYGUI::preferenceChanged()")
    get_aster_gui().preferenceChanged(section, name)


def windows():
    """
    Specify standard SALOME dock windows needed by module and their
    positions.

    Returns:
        dict: {window:position} where `window` is SalomePyQt.WindowType
        enumerator and `position` is Qt.DockWidgetArea enumerator
    """
    debug_message("ASTERSTUDYGUI::windows()")
    # we do not need any standard SALOME dock windows;
    # below line is needed to prevent default implementation.
    return {}


def views():
    """
    Specify view windows needed by module.

    Returns:
        list: result contains views' type names
    """
    debug_message("ASTERSTUDYGUI::views()")
    # we want our own view window to be automatically pop up
    # when module is activated.
    return [get_aster_view_type()]


def saveFiles(*args):
    """
    Save module data to the files; returns file names.

    The function saves the module data to the file (or files) in a
    temporary directory specified as a parameter and returns file name
    (or file names) in which module data is saved.

    Depending on SALOME version, list of parameters may contain one or
    two items:

    - directory (str): A directory to store data files. Note: this can
      be not a final study destination folder but a temporary directly,
      depending on used save mode (single-file or multi-file).

    - url (str): Actual study URL (the final study destination). Note:
      this parameter is provided for information purposes only!

    Arguments:
        *args: Positional parameters.

    Returns:
        list: names of files in which data is saved
    """
    directory = args[0] if len(args) > 0 else ""
    url = args[1] if len(args) > 1 else ""
    debug_message("ASTERSTUDYGUI::saveFiles()", directory, url)
    return get_aster_gui().save(directory, url)


def openFiles(*args):
    """
    Load data from the files; return result status.

    The function restores module data from the file(s) specified as a
    parameter; returns *True* in case of success or *False* otherwise.

    Depending on SALOME version, list of parameters may contain one or
    two items:

    - files (list[str]): Data files in which module data is stored.
      Note: first element of this list is a directory name. File names
      are normally specified as relative to this directory.

    - url (str): Actual study URL (the original study file path). Note:
      this parameter is provided for information purposes only!

    Arguments:
        *args: Positional parameters.

    Returns:
        bool: *True* in case of success; *False* otherwise
    """
    files = args[0] if len(args) > 0 else ""
    url = args[1] if len(args) > 1 else ""
    debug_message("ASTERSTUDYGUI::openFiles()", files, url)
    return get_aster_gui().load(files, url)


# pragma pylint: disable=unused-argument
def closeStudy(study_id):
    """
    Process study closure.

    The function is called when SALOME study is closed, to perform
    needed clean-up of the data model.

    Arguments:
        study_id: study identifier
    """
    debug_message("ASTERSTUDYGUI::closeStudy()")
    get_aster_gui().close()
