# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Validity report dialog
----------------------

The module implements a dialog for validity report.

"""

from __future__ import unicode_literals

from PyQt5 import Qt as Q

from . import Context
from ..common import load_pixmap, translate
from ..datamodel.command import Comment
from ..datamodel.general import Validity
from .datasettings.model import get_object_name, get_object_type
from .widgets import Dialog

# note: the following pragma is added to prevent pylint complaining
#       about functions that follow Qt naming conventions;
#       it should go after all global functions
# pragma pylint: disable=invalid-name

MARGIN = 5
SPACING = 5

ROW_HEIGHT = 20
PIXMAP_MARGIN = 5


class ValidityReportDialog(Dialog):
    """Validity report dialog."""

    def __init__(self, astergui, stage, parent=None):
        """
        Create dialog.

        Arguments:
            astergui (AsterGui): Parent AsterGui instance.
            stage (Stage): Instance of stage object.
            parent (Optional[QWidget]): Parent widget. Defaults to *None*.
        """
        super(ValidityReportDialog, self).__init__(parent)

        # Main parameters
        title = translate("ValidityReportDialog",
                          "Validity report for {}").format(stage.name)
        self.setWindowTitle(title)
        self.setModal(True)

        self._astergui = astergui

        ValidityData = [(1, Validity.Syntaxic, "Syntaxic"),
                        (2, Validity.Dependency, "Dependency"),
                        (3, Validity.Naming, "Naming")]

        # Table of commands
        tableFrame = Q.QFrame(self.frame())
        tableFrame.setFrameStyle(Q.QFrame.Box | Q.QFrame.Sunken)

        self._table = Q.QTableWidget(tableFrame)

        headerLabels = ["Command"]
        for data in ValidityData:
            headerLabels.append(data[2])
        self._table.setColumnCount(len(headerLabels))
        self._table.setHorizontalHeaderLabels(headerLabels)

        self._table.horizontalHeader().setSectionResizeMode(
            0, Q.QHeaderView.Stretch)
        for data in ValidityData:
            self._table.setColumnWidth(data[0], 100)

        self._table.verticalHeader().hide()

        self._table.setEditTriggers(Q.QAbstractItemView.NoEditTriggers)
        self._table.setSelectionBehavior(Q.QAbstractItemView.SelectRows)

        pixmapError = load_pixmap("as_pic_msglevel_error.png")
        pixmapError = pixmapError.scaled(ROW_HEIGHT - PIXMAP_MARGIN,
                                         ROW_HEIGHT - PIXMAP_MARGIN,
                                         Q.Qt.KeepAspectRatio,
                                         Q.Qt.SmoothTransformation)

        self._commands = []
        commands = stage.sorted_commands
        for command in commands:
            if not isinstance(command, Comment):
                self._commands.append(command)

        self._table.setRowCount(len(self._commands))

        for row, command in enumerate(self._commands):
            self._table.setRowHeight(row, ROW_HEIGHT)

            command_name = ""
            try:
                command_name = get_object_name(command)
                command_type = get_object_type(command)
                if command_type:
                    command_name += " ({})".format(command_type)
            except RuntimeError:
                # for batch mode
                command_name = command.name

            item = Q.QTableWidgetItem()
            item.setText(command_name)
            self._table.setItem(row, 0, item)

            check = command.check()

            for data in ValidityData:
                if check & data[1]:
                    label = Q.QLabel()
                    label.setAlignment(Q.Qt.AlignCenter)
                    label.setPixmap(pixmapError)
                    self._table.setCellWidget(row, data[0], label)

        showValidCommands = Q.QCheckBox(translate("ValidityReportDialog",
                                                  "Show valid commands"),
                                        tableFrame)
        showValidCommands.setChecked(True)

        # Layout
        tableFrameLayout = Q.QVBoxLayout(tableFrame)
        tableFrameLayout.setContentsMargins(MARGIN, MARGIN, MARGIN, MARGIN)
        tableFrameLayout.setSpacing(SPACING)
        tableFrameLayout.addWidget(self._table)
        tableFrameLayout.addWidget(showValidCommands)

        self.setStandardButtons(Q.QDialogButtonBox.Close)

        self.frame().layout().addWidget(tableFrame)

        # Connections
        self._table.cellDoubleClicked.connect(self.cellDoubleClicked)

        showValidCommands.toggled.connect(self.showValidCommands)

        self.button(Q.QDialogButtonBox.Close).clicked.connect(self.close)

        # Initialization
        self.resize(600, 500)

    @Q.pyqtSlot(int, int)
    def cellDoubleClicked(self, row, _):
        """
        Slot called when the table cell is double clicked.

        Arguments:
            row (int): Table row.
            column (int): Table column.
        """
        if row < 0 or row >= len(self._commands):
            return

        command = self._commands[row]
        if command:
            self._astergui.update(autoSelect=command,
                                  context=Context.DataSettings)
        self.close()

    @Q.pyqtSlot(bool)
    def showValidCommands(self, state):
        """
        Slot called when the option 'Show valid commands' is toggled on/off.

        Arguments:
            state (bool): Toggle state.
        """
        for row in range(self._table.rowCount()):
            isHidden = False
            if not state:
                isError = False
                for column in range(1, self._table.columnCount()):
                    isError |= (self._table.cellWidget(row, column) != None)
                isHidden = not isError
            self._table.setRowHidden(row, isHidden)
