# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Stage text editor
-----------------

The module implements an editor for text stage.

"""

from __future__ import unicode_literals

from PyQt5 import Qt as Q

from ..common import enable_autocopy, load_pixmap, translate
from ..datamodel import CATA
from .controller import WidgetController
from .editionwidget import EditionWidget
from .widgets import ConceptsEditor, TextEditor

# note: the following pragma is added to prevent pylint complaining
#       about functions that follow Qt naming conventions;
#       it should go after all global functions
# pragma pylint: disable=invalid-name


class StageTextEditor(EditionWidget, WidgetController):
    """Stage text editor widget."""

    def __init__(self, stage, astergui, parent=None, **kwargs):
        """
        Create editor.

        Arguments:
            stage (Stage): Stage to edit.
            astergui (AsterGui): AsterGui instance.
            parent (Optional[QWidget]): Parent widget. Defaults to
                *None*.
            **kwargs: Keyword arguments.
        """
        super(StageTextEditor, self).__init__(parent=parent,
                                              name=translate("StageTextEditor",
                                                             "Edit stage"),
                                              astergui=astergui, **kwargs)

        self.stage = stage
        self.prev_state = stage.get_text()

        title = translate("StageTextEditor", "Edit stage '{}'")
        self.setWindowTitle(title.format(stage.name))
        self.setPixmap(load_pixmap("as_pic_edit_stage.png"))

        self.editor = TextEditor(self)
        commands = [j for i in [CATA.get_category(i) for i in \
                                    CATA.get_categories("showall")] for j in i]
        self.editor.setKeywords(commands, 0, Q.QColor("#ff0000"))
        self.editor.setText(self.prev_state)
        self.editor.textChanged.connect(self.updateButtonStatus)
        self.setFocusProxy(self.editor)
        astergui.preferencesChanged.connect(self.editor.updateSettings)
        self.editor.updateSettings(self.astergui().preferencesMgr())

        self.concepts_editor = ConceptsEditor(stage, self)
        self.concepts_editor.modified.connect(self.updateButtonStatus)

        self.splitter = Q.QSplitter(Q.Qt.Horizontal, self)
        self.splitter.addWidget(self.editor)
        self.splitter.addWidget(self.concepts_editor)
        self.splitter.setStretchFactor(0, 3)
        self.splitter.setStretchFactor(1, 2)

        v_layout = Q.QVBoxLayout(self)
        v_layout.setContentsMargins(0, 0, 0, 0)
        v_layout.setSpacing(5)
        v_layout.addWidget(self.splitter)

    def setTextEditorVisible(self, state):
        """
        Set visibility of the text editor.

        Arguments:
            state (bool): Visibility state.
        """
        self.editor.setVisible(state)

    def isButtonEnabled(self, button):
        """
        Redefined from *EditionWidget* class.
        """
        result = True
        if button in [Q.QDialogButtonBox.Ok, Q.QDialogButtonBox.Apply]:
            result = self.stage and (
                (self.editor.isVisible() and \
                 self.editor.text() != self.prev_state) or \
                self.concepts_editor.isApplyAllowed())
        return result

    def applyChanges(self):
        """
        Redefined from *EditionWidget* class.
        """
        with enable_autocopy(self.astergui().study().activeCase):
            try:
                self.concepts_editor.applyChanges()
                self.prev_state = self.editor.text()
                self.stage.set_text(self.prev_state)
                msg = translate("StageTextEditor", "Edit stage")
                self.astergui().study().commit(msg)
                self.astergui().update()
            except Exception: # pragma pylint: disable=broad-except
                self.astergui().study().revert()

    def setReadOnly(self, on):
        """
        Redefined from *EditionWidget* class.
        """
        super(StageTextEditor, self).setReadOnly(on)
        self.editor.setReadOnly(on)
        self.concepts_editor.setReadOnly(on)

    def controllerStart(self):
        """Reimplemented from WidgetController."""
        if super(StageTextEditor, self).controllerStart():
            self.astergui().workSpace().setMeshViewEnabled(False)

    def controllerStop(self):
        """Reimplemented from WidgetController."""
        self.astergui().workSpace().setMeshViewEnabled(True)
        super(StageTextEditor, self).controllerStop()

    def updateTranslations(self):
        """
        Update translations in GUI elements.
        """
        self.concepts_editor.updateTranslations()
