# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Show All panel
--------------

This module implements `Show All` panel that allows the user adding the
command from the catalogue to the study.

"""

from __future__ import unicode_literals

from PyQt5 import Qt as Q

from . import (Context, NodeType, UrlHandler, get_node_type,
               translate_category, translate_command)
from ..common import (bold, connect, enable_autocopy, italic, load_icon,
                      load_pixmap, preformat, translate)
from ..datamodel import CATA
from .behavior import behavior
from .controller import WidgetController
from .editionwidget import EditionWidget
from .widgets import CategoryView, FilterPanel

__all__ = ["ShowAllPanel"]

# note: the following pragma is added to prevent pylint complaining
#       about functions that follow Qt naming conventions;
#       it should go after all global functions
# pragma pylint: disable=invalid-name


class ShowAllPanel(EditionWidget, WidgetController):
    """`Show All` panel."""

    def __init__(self, astergui, parent=None):
        """
        Create `Show all` edition panel.

        Arguments:
            astergui (AsterGui): Parent AsterGui instance.
            parent (Optional[QWidget]): Parent widget. Defaults to
                *None*.
        """
        super(ShowAllPanel, self).__init__(parent=parent,
                                           name=translate("ShowAllPanel",
                                                          "Show all commands"),
                                           astergui=astergui)
        self.setWindowTitle(translate("ShowAllPanel", "Add command"))
        self.setPixmap(load_pixmap("as_pic_new_command.png"))
        v_layout = Q.QVBoxLayout(self)
        v_layout.setContentsMargins(0, 0, 0, 0)
        self.panel = Panel(self)
        description = Q.QGroupBox(self)
        description.setTitle(translate("ShowAllPanel", "Command description"))
        desc_layout = Q.QVBoxLayout(description)
        self.info = Q.QLabel(description)
        self.info.setWordWrap(True)
        self.info.setOpenExternalLinks(True)
        desc_layout.addWidget(self.info)
        self.doc_url = ""
        self.doc_button = Q.QPushButton(load_icon("as_pic_help.png"),
                                        " " + translate("AsterStudy",
                                                        "Documentation"),
                                        self)
        self.doc_button.setEnabled(False)
        self.doc_button.clicked.connect(self._docButtonClicked)
        desc_layout.addWidget(self.doc_button)

        favoritesMgr = astergui.favoritesMgr()
        favorites_category = favoritesMgr.name()
        categories = [favorites_category]
        categories.extend(CATA.get_categories("showall"))
        for category in categories:
            # get commands from category
            items = []
            if category == favorites_category:
                items = favoritesMgr.commands()
            else:
                items = CATA.get_category(category)
                if not items:
                    continue
            # get translation for category's title
            view = CategoryView(translate_category(category),
                                parent=self.panel)
            view.category = category
            for item in items:
                subcategory = CATA.get_command_subcategory(item)
                if subcategory:
                    subcategory = translate_category(subcategory)
                # get translation for command
                title = translate_command(item)
                if title != item:
                    title = title + " ({})".format(item)
                view.addItem(title, item, subcategory)
            self.panel.addWidget(view)
            connect(view.selected, self._selectionChanged)
            connect(view.doubleClicked, self._doubleClicked)
            if category == favorites_category:
                view.setVisible(len(items) > 0)
        v_layout.addWidget(self.panel)
        v_layout.addWidget(description)
        self.panel.installEventFilter(self)
        self.setFocusProxy(self.panel)
        astergui.favoritesChanged.connect(self.updateTranslations)

    def defaultButton(self):
        """
        Get button to be used for default action.

        Default implementation returns *None*.
        """
        return Q.QDialogButtonBox.Ok

    def isButtonEnabled(self, button):
        """
        Redefined from *EditionWidget* class.
        """
        result = True
        if button in [Q.QDialogButtonBox.Ok, Q.QDialogButtonBox.Apply]:
            result = self._selected() is not None
        return result

    def accept(self):
        """Redefined from *EditionWidget* class."""
        if not self.astergui().canAddCommand():
            message = translate("AsterStudy",
                                "Please select graphical stage "
                                "to add a command")
            Q.QMessageBox.critical(self, "AsterStudy", message)
            return False
        return True

    def applyChanges(self):
        """
        Redefined from *EditionWidget* class.
        """
        command = self._selected()
        if command:
            cmdobj = None
            selected = self.astergui().selected(Context.DataSettings)
            stage = self.astergui().study().node(selected[0])
            if get_node_type(stage) not in (NodeType.Stage,):
                stage = stage.stage
            if stage is not None:
                try:
                    cmdobj = stage.add_command(command)
                except Exception: # pragma pylint: disable=broad-except
                    cmdobj = None
            if cmdobj is not None:
                with enable_autocopy(self.astergui().study().activeCase):
                    self.astergui().study().commit(translate("AsterStudy",
                                                             "Add command"))
                    self.astergui().update(autoSelect=cmdobj,
                                           context=Context.DataSettings)
                    msg = translate("AsterStudy",
                                    "Command with type '{}' "
                                    "successfully added")
                    msg = msg.format(command)
                    self.astergui().showMessage(msg)
            else:
                self.astergui().study().revert()

    def postClose(self, button):
        """
        Redefined from *EditionWidget* class.

        Activates automatic edition of the just added command if *OK*
        button is clicked.
        """
        if button in (Q.QDialogButtonBox.Ok,):
            if behavior().auto_edit:
                self.astergui().edit()

    @Q.pyqtSlot()
    def addToFavorites(self):
        """Add the selected command to the Favorites category."""
        command = self._selected()
        if command:
            self.astergui().favoritesMgr().addCommand(command)
            self.astergui().updateFavorites()

    @Q.pyqtSlot()
    def removeFromFavorites(self):
        """Remove the selected command from the Favorites category."""
        command = self._selected()
        if command:
            self.astergui().favoritesMgr().removeCommand(command)
            self.astergui().updateFavorites()

    @Q.pyqtSlot()
    def expandAll(self):
        """Expand all categories."""
        for view in self.panel.widgets():
            view.expand()

    @Q.pyqtSlot()
    def collapseAll(self):
        """Collapse all categories."""
        for view in self.panel.widgets():
            view.collapse()

    def updateTranslations(self):
        """
        Update translations in GUI elements.
        """
        views = self.panel.widgets()
        for view in views:
            items = []
            favoritesMgr = self.astergui().favoritesMgr()
            if view.category == favoritesMgr.name():
                items = favoritesMgr.commands()
                view.setVisible(len(items) > 0)
            else:
                items = CATA.get_category(view.category)
            view.clear()
            for item in items:
                subcategory = CATA.get_command_subcategory(item)
                if subcategory:
                    subcategory = translate_category(subcategory)
                title = translate_command(item)
                if title != item:
                    title = title + " ({})".format(item)
                view.addItem(title, item, subcategory)
        self.panel.applyFilter()

    def eventFilter(self, obj, event):
        """
        Filter events if this object has been installed as an event
        filter for the watched object.

        Shows dedicated context menu for categories panel.

        Arguments:
            obj (QObject): Watched object.
            event (QEvent): Event being processed.

        Returns:
            bool: *True* if event should be filtered out (i.e. if
            further processing should be stopped); *False* otherwise.
        """
        if obj == self.panel and event.type() == Q.QEvent.ContextMenu:
            menu = Q.QMenu()
            command = self._selected()
            if command:
                menu.addAction(translate("ShowAllPanel", "Add to Study"),
                               self._addToStudy)
                menu.addSeparator()
                favoritesMgr = self.astergui().favoritesMgr()
                if favoritesMgr.hasCommand(command):
                    menu.addAction(translate("ShowAllPanel",
                                             "Remove from Favorites"),
                                   self.removeFromFavorites)
                else:
                    menu.addAction(translate("ShowAllPanel",
                                             "Add to Favorites"),
                                   self.addToFavorites)
                menu.addSeparator()
            menu.addAction(translate("ShowAllPanel", "Expand all"),
                           self.expandAll)
            menu.addAction(translate("ShowAllPanel", "Collapse all"),
                           self.collapseAll)
            event.accept()
            menu.exec_(self.mapToGlobal(event.pos()))
        # pragma pylint: disable=no-member
        return False

    @Q.pyqtSlot(str)
    def _selectionChanged(self, command):
        """
        Called when a command is selected in any category.

        Arguments:
            command (str): Command name.
        """
        sender_view = self.sender()
        for view in self.panel.widgets():
            if view != sender_view:
                view.clearSelection()
        if command:
            translation = translate_command(command)
            description = CATA.get_command_docstring(command)
            self.doc_url = self.astergui().doc_url(command)
            if translation != command:
                text = italic(translation) + " ({})".format(bold(command))
            else:
                text = bold(command)
            text = preformat(text)
            if description:
                text = text + "<br/>" + description
            self.info.setText(text)
            self.doc_button.setEnabled(True)
        else:
            self.doc_url = ""
            self.info.setText("")
            self.doc_button.setEnabled(False)
        self.updateButtonStatus()

    @Q.pyqtSlot()
    def _addToStudy(self):
        """
        'Add to Study' context menu action's slot.
        """
        if self.isButtonEnabled(Q.QDialogButtonBox.Apply):
            self.perform(Q.QDialogButtonBox.Apply)

    @Q.pyqtSlot(str)
    def _doubleClicked(self):
        """
        Called when a command is double-clicked in any category.

        Arguments:
            command (str): Command name.
        """
        if self.isButtonEnabled(Q.QDialogButtonBox.Ok):
            self.perform(Q.QDialogButtonBox.Ok)

    @Q.pyqtSlot(bool)
    def _docButtonClicked(self):
        UrlHandler().open_url(self.doc_url)

    def _selected(self):
        """
        Get selected command (*None* if there is no selection).

        Returns:
            str: Selected command.
        """
        for view in self.panel.widgets():
            if view.selection():
                return view.selection()
        return None


class Panel(FilterPanel):
    """
    Custom filter panel to automatically select first item that
    satisfies search criterion.
    """

    @Q.pyqtSlot(str)
    def filter(self, text):
        """Redefined from *FilterPanel* class."""
        super(Panel, self).filter(text)
        for widget in self.widgets():
            if not isinstance(widget, CategoryView):
                continue
            if widget.visibleCount() > 0:
                widget.expand()
                widget.select(0)
                break
