# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Run panel
---------

The module implements panel with run parameters for AsterStudy GUI.
See `RunPanel` class for more details.

"""

from __future__ import unicode_literals

import os.path as osp

from PyQt5 import Qt as Q

from . import Context, Entity, NodeType, check_selection, get_node_type
from ..common import is_localhost, load_icon, translate, wait_cursor
from ..datamodel.engine import (default_parameters, serverinfos_factory,
                                version_ismpi)

# note: the following pragma is added to prevent pylint complaining
#       about functions that follow Qt naming conventions;
#       it should go after all global functions
# pragma pylint: disable=invalid-name

# The following global settings affects how run panel is disabled
# when Run case is selected
DISABLE_ALL_TABS = True

# pragma pylint: disable=too-many-instance-attributes
class RunPanel(Q.QWidget):
    """
    Panel with run parameters.
    """
    parametersChanged = Q.pyqtSignal()

    class TimeEdit(Q.QWidget):
        """
        Class that represents the input time field as tree
        separate parts - hours, minutes and seconds
        """

        def __init__(self, parent=None):
            super(RunPanel.TimeEdit, self).__init__(parent)

            base = Q.QHBoxLayout(self)
            base.setContentsMargins(0, 0, 0, 0)

            self._hour = Q.QSpinBox(self)
            self._hour.setMinimum(0)
            self._hour.setMaximum(10000)
            self._hour.setSizePolicy(Q.QSizePolicy(Q.QSizePolicy.Expanding,
                                                   Q.QSizePolicy.Preferred))
            base.addWidget(self._hour)

            base.addWidget(Q.QLabel(':', self))

            self._minute = Q.QSpinBox(self)
            self._minute.setMinimum(0)
            self._minute.setMaximum(60)
            self._minute.setSizePolicy(Q.QSizePolicy(Q.QSizePolicy.Expanding,
                                                     Q.QSizePolicy.Preferred))
            base.addWidget(self._minute)

            base.addWidget(Q.QLabel(':', self))

            self._second = Q.QSpinBox(self)
            self._second.setMinimum(0)
            self._second.setMaximum(60)
            self._second.setSizePolicy(Q.QSizePolicy(Q.QSizePolicy.Expanding,
                                                     Q.QSizePolicy.Preferred))
            base.addWidget(self._second)

        def time(self):
            """
            Gets the current time

            Returns:
                str: String in format hh:mm:ss
            """
            return "{0:02d}:{1:02d}:{2:02d}".format(self._hour.value(),
                                                    self._minute.value(),
                                                    self._second.value())

        def setTime(self, timestr):
            """
            Sets the time into edit

            Arguments:
                timestr (str): String with time in format hh:mm:ss
            """
            timelist = timestr.split(':')
            times = []
            for i in timelist:
                num = 0
                try:
                    num = int(i)
                except ValueError:
                    num = 0
                times.append(num)

            while len(times) < 3:
                times.append(0)

            self._hour.setValue(times[0])
            self._minute.setValue(times[1])
            self._second.setValue(times[2])

        def isReadOnly(self):
            """
            Gets the read only state
            """
            return self._hour.isReadOnly() and self._minute.isReadOnly() and \
                self._second.isReadOnly()

        def setReadOnly(self, state):
            """
            Sets the read only state
            """
            self._hour.setReadOnly(state)
            self._minute.setReadOnly(state)
            self._second.setReadOnly(state)


    class PathEdit(Q.QWidget):
        """
        Class that represents the input path field with browse button.
        """
        pathChanged = Q.pyqtSignal(str)

        def __init__(self, parent=None):
            super(RunPanel.PathEdit, self).__init__(parent)

            self._path = Q.QLineEdit(self)
            self._path.setObjectName('RunPanel.path')
            self._path.setCompleter(Q.QCompleter(Q.QDirModel(self._path),
                                                 self._path))
            self._path.textChanged.connect(self.pathChanged)

            self._browse = Q.QToolButton(self)
            self._browse.setText("...")
            self._browse.clicked.connect(self._browseClicked)

            base = Q.QHBoxLayout(self)
            base.setContentsMargins(0, 0, 0, 0)
            base.addWidget(self._path)
            base.addWidget(self._browse)

        def path(self):
            """
            Gets the current path

            Returns:
                str: Path string
            """
            return self._path.text()

        def setPath(self, value):
            """
            Sets the current path

            Arguments:
                value (str): Path string
            """
            self._path.setText(value)

        def isReadOnly(self):
            """
            Gets the read only state.
            """
            return self._path.isReadOnly()

        def setReadOnly(self, state):
            """
            Sets the read only state.
            """
            self._path.setReadOnly(state)

        def _browseClicked(self):
            """
            Invoked when 'Browse' button clicked.
            """
            if not self.isReadOnly():
                path = self.path()
                caption = translate("RunPanel", "Directory")
                sel_path = Q.QFileDialog.getExistingDirectory(self,
                                                              caption, path)
                if sel_path:
                    sel_path = Q.QDir.toNativeSeparators(sel_path)
                    self.setPath(sel_path)

    class RemotePathEdit(PathEdit):
        """
        Input path field with browse button and checkbox.
        """
        remoteStateChanged = Q.pyqtSignal(bool)

        def __init__(self, parent=None):
            """
            Calls parent initializer and inserts the checkbox.
            """
            super(RunPanel.RemotePathEdit, self).__init__(parent)

            self._checkbox = Q.QCheckBox(self)
            self._checkbox.stateChanged.connect(self._checkboxStateChanged)

            base = self.layout()
            base.insertWidget(0, self._checkbox)

            # Waiting for a portable option to browse remote files,
            # Hide the browse button
            base.removeWidget(self._browse)
            self._browse.hide()

            self._checkbox.stateChanged.emit(self._checkbox.checkState())

        def state(self):
            """
            State of the checkbox.
            """
            return True if self._checkbox.checkState() == Q.Qt.Checked \
                   else False

        def _checkboxStateChanged(self, state):
            """
            Called when checkbox is ticked or unticked.

            Note:
                Ticking enables text edit and browsing.
                Wrapper that emits `remoteStateChanged`.
            """
            bool_state = True if state == Q.Qt.Checked else False

            self._path.setEnabled(bool_state)

            self.remoteStateChanged.emit(bool_state)

    # keep in memory last parameters used per session
    # stored at runParameters getter call
    _last_params = default_parameters()

    @classmethod
    def getCachedParams(cls):
        """Return the dict used for this kind of version.

        Return:
            dict: Dict that store the last parameters used.
        """
        return cls._last_params

    @classmethod
    def setCachedParams(cls, params):
        """Register the dict used for this kind of version.

        Arguments:
            params (dict): Dict of the last parameters used.
        """
        cls._last_params = params

    def __init__(self, astergui, parent=None):
        super(RunPanel, self).__init__(parent)
        self._astergui = astergui
        self._case = None

        base = Q.QVBoxLayout(self)
        base.setContentsMargins(5, 5, 5, 5)

        self.tab_widget = Q.QTabWidget(self)
        base.addWidget(self.tab_widget)

        basic_tab = Q.QWidget(self)
        basic_grid = Q.QGridLayout(basic_tab)
        advan_tab = Q.QWidget(self)
        advan_grid = Q.QGridLayout(advan_tab)

        # interface to server informations
        self._infos = serverinfos_factory()

        # Basic tab

        # Run case name
        self._casename = Q.QLineEdit(self)
        self._addWidget(basic_grid,
                        translate("RunPanel", "Case name"), self._casename)

        # Memory
        self._memory = Q.QLineEdit(self)
        self._memory.setValidator(Q.QIntValidator(self._memory))
        self._memory.validator().setBottom(0)
        self._addWidget(basic_grid,
                        translate("RunPanel", "Memory"), self._memory)

        # Time
        self._time = RunPanel.TimeEdit(self)
        self._addWidget(basic_grid,
                        translate("RunPanel", "Time"), self._time)

        # Server
        self._server = Q.QComboBox(self)
        self._server.addItems(self._infos.available_servers)
        updbutton = Q.QToolButton(self)
        updbutton.setIcon(load_icon("as_pic_update.png"))

        srvcont = Q.QWidget(self)
        srvbase = Q.QHBoxLayout(srvcont)
        srvbase.setContentsMargins(0, 0, 0, 0)
        srvbase.addWidget(self._server)
        srvbase.addWidget(updbutton)

        self._addWidget(basic_grid,
                        translate("RunPanel", "Run servers"), srvcont)

        # code_aster version
        self._history_version = None
        self._version = Q.QComboBox(self)
        self._addWidget(basic_grid,
                        translate("RunPanel",
                                  "Version of code_aster"), self._version)

        # MPI CPU
        self._mpicpu = Q.QLineEdit(self)
        self._mpicpu.setValidator(Q.QIntValidator(1, 8192, self._mpicpu))
        self._addWidget(basic_grid,
                        translate("RunPanel",
                                  "Number of MPI CPU"), self._mpicpu)

        # User description
        self._descr = Q.QTextEdit(self)
        self._addWidget(basic_grid,
                        translate("RunPanel",
                                  "User description"), self._descr)

        # Advanced tab

        # Run mode
        self._mode = Q.QComboBox(self)
        self._addWidget(advan_grid,
                        translate("RunPanel", "Run mode"), self._mode)

        # Exec mode
        self._execmode = Q.QComboBox(self)
        self._addWidget(advan_grid,
                        translate("RunPanel", "Execution mode"),
                        self._execmode)

        # History folder
        self._folder = RunPanel.PathEdit(self)
        self._addWidget(advan_grid,
                        translate("RunPanel",
                                  "History folder"), self._folder)
        self._folder.pathChanged.connect(self.parametersChanged)

        # Remote folder (where to store result databases)
        self._remote_folder = RunPanel.RemotePathEdit(self)
        self._addWidget(advan_grid,
                        translate("RunPanel",
                                  "Remote databases"), self._remote_folder)
        self._remote_folder.remoteStateChanged.connect(self.parametersChanged)
        self._remote_folder.pathChanged.connect(self.parametersChanged)

        # Number of nodes
        self._nodes = Q.QLineEdit(self)
        self._nodes.setValidator(Q.QIntValidator(1, 8192, self._nodes))
        self._addWidget(advan_grid,
                        translate("RunPanel",
                                  "Number of MPI nodes"), self._nodes)

        # Number of threads
        self._threads = Q.QLineEdit(self)
        self._threads.setValidator(Q.QIntValidator(0, 8192, self._threads))
        self._addWidget(advan_grid,
                        translate("RunPanel",
                                  "Number of threads"), self._threads)

        # Server partition
        self._srvpart = Q.QLineEdit(self)
        self._addWidget(advan_grid,
                        translate("RunPanel", "Server partition"),
                        self._srvpart)

        # Priority QOS
        self._qos = Q.QLineEdit(self)
        self._addWidget(advan_grid,
                        translate("RunPanel", "QOS"), self._qos)

        #---------------------------------------------------------------------
        self.tab_widget.addTab(basic_tab, translate("RunPanel", "Basic"))
        self.tab_widget.addTab(advan_tab, translate("RunPanel", "Advanced"))

        basic_grid.setRowStretch(basic_grid.rowCount(), 1)
        advan_grid.setRowStretch(advan_grid.rowCount(), 1)

        #---------------------------------------------------------------------
        updbutton.clicked.connect(self._serverUpdate)
        self._server.activated.connect(self._serverActivated)
        self._version.activated.connect(self._versionActivated)
        self._mpicpu.textEdited.connect(self._mpicpuEdited)
        self._nodes.textEdited.connect(self._nodesEdited)

        self._serverActivated()
        self._versionActivated()

        self.setWindowTitle(translate("RunPanel", "Run parameters"))
        self._updateState()

    def selection(self):
        """
        Gets the currently selected objects.

        Returns:
            list: List of selected objects: cases.
        """
        case = self.case()
        return [Entity(case.uid, get_node_type(case))] \
            if case is not None else []

    def setSelection(self, objs):
        """
        Sets the selection, i.e. select given objects.
        Other objects will be unselected.

        Arguments:
            objs (list): List of objects that should be selected.
        """
        case = None
        if check_selection(objs, size=1, typeid=NodeType.Case):
            case = self._astergui.study().node(objs[0])

        self.setCase(case)

    def update(self):
        """
        Updates the panel
        """
        study = self._astergui.study()
        hasfolder = study.history.folder is not None and \
            len(study.history.folder) > 0
        hasremote = study.history.remote_folder is not None and \
            len(study.history.remote_folder) > 0

        path = ''
        if hasfolder:
            path = study.history.folder
        else:
            path = study.url()
            if path:
                path = osp.splitext(path)[0] + '_Files'

        params = self.getCachedParams() \
            if self.case() != study.history.current_case \
            else self.runParameters()

        last = self._lastStage()
        if last is not None:
            previous = last.result.job.asdict()
            previous_version = previous.get('version')
            if previous_version:
                params['version'] = previous_version

        params['folder'] = path
        params['casename'] = self._generateName()
        if self.case() != study.history.current_case:
            self.setCachedParams(params)
        else:
            self.setRunParameters(params)

        self.setFolderEnabled(False)
        self.setRemoteFolderEnabled(not hasremote and not self.isLocal())
        self.setHistoryVersion(study.history.version)

    def case(self):
        """
        Gets the case
        """
        return self._case

    def setCase(self, case):
        """
        Sets the case
        """
        if self._case != case:
            if self._case == self._astergui.study().history.current_case:
                RunPanel.setCachedParams(self.runParameters())
            self._case = case
            self._updateState()

    def setReadOnly(self, state):
        """
        Sets the read only state to all controls.
        """
        if DISABLE_ALL_TABS:
            for index in range(self.tab_widget.count()):
                self.tab_widget.widget(index).setDisabled(state)
        else:
            self._casename.setReadOnly(state)
            self._memory.setReadOnly(state)
            self._time.setReadOnly(state)
            self._server.setEnabled(not state)
            self._version.setEnabled(not state)
            self._mpicpu.setReadOnly(state)
            self._descr.setReadOnly(state)
            self._mode.setEnabled(not state)
            self._execmode.setEnabled(not state)
            self._folder.setReadOnly(state)
            self._remote_folder.setReadOnly(state)
            self._nodes.setReadOnly(state)
            self._threads.setReadOnly(state)
            self._srvpart.setReadOnly(state)
            self._qos.setReadOnly(state)

    def caseName(self):
        """
        Gets the run case name.

        Returns:
            str: Run case name.
        """
        return self._casename.text()

    def setCaseName(self, name):
        """
        Sets the run case name.

        Arguments:
            name (str): Run case name.
        """
        self._casename.setText(name)

    def memory(self):
        """
        Gets the memory quantity.

        Returns:
            int: Memory quantity.
        """
        val = self._memory.text()
        return int(val) if len(val) > 0 else 0

    def setMemory(self, mem):
        """
        Sets the memory quantity.

        Arguments:
            mem (int): Memory quantity.
        """
        self._memory.setText(str(mem))

    def time(self):
        """
        Gets the running time.

        Returns:
            str: Running time.
        """
        return self._time.time()

    def setTime(self, timeval):
        """
        Sets the running time.

        Arguments:
            timeval (str): Running time.
        """
        self._time.setTime(timeval)

    def server(self):
        """
        Gets the currently selected run server.

        Returns:
            str: Run server name.
        """
        return self._server.currentText()

    def setServer(self, serv):
        """
        Sets the current run server.

        Arguments:
            serv (str): Run server name.
        """
        self._server.setCurrentIndex(self._server.findText(serv))
        self._serverActivated()

    def codeAsterVersion(self):
        """
        Gets the current code_aster version.

        Returns:
            str: code_aster version.
        """
        return self._version.currentText()

    def setCodeAsterVersion(self, ver):
        """
        Sets the current code_aster version.

        Arguments:
            ver (str): code_aster version.
        """
        idx = self._version.findText(ver)
        if idx >= 0:
            self._version.setCurrentIndex(idx)
        self._versionActivated()

    def execMode(self):
        """
        Gets the execution mode.

        Returns:
            str: Debug mode.
        """
        return self._execmode.currentText()

    def setExecMode(self, value):
        """
        Sets the execution mode.

        Arguments:
            value (str): Debug mode.
        """
        self._execmode.setCurrentIndex(self._execmode.findText(value))

    def mode(self):
        """
        Gets the running mode.

        Returns:
            int: Running mode.
        """
        return self._mode.currentText()

    def setMode(self, mode):
        """
        Sets the running mode.

        Arguments:
            mode (int): Running mode.
        """
        idx = self._mode.findText(mode)
        self._mode.setCurrentIndex(max(idx, 0))

    def mpiCpu(self):
        """
        Gets the number of MPI CPU's.

        Returns:
            int: number of MPI CPU's.
        """
        val = self._mpicpu.text()
        return int(val) if len(val) > 0 else 0

    def setMpiCpu(self, cpu):
        """
        Sets the number of MPI CPU's.

        Arguments:
            cpu (int): Number of MPI CPU's.
        """
        self._mpicpu.setText(str(cpu))

    def threads(self):
        """
        Gets the number of threads.

        Returns:
            int: number of threads.
        """
        val = self._threads.text()
        return int(val) if len(val) > 0 else 0

    def setThreads(self, num):
        """
        Sets the number of threads.

        Arguments:
            num (int): Number of threads.
        """
        self._threads.setText(str(num))

    def nodes(self):
        """
        Gets the number of nodes.

        Returns:
            int: number of nodes.
        """
        val = self._nodes.text()
        return int(val) if len(val) > 0 else 0

    def setNodes(self, num):
        """
        Sets the number of nodes.

        Arguments:
            num (int): Number of nodes.
        """
        self._nodes.setText(str(num))

    def userDescription(self):
        """
        Gets the user description.

        Returns:
            str: User description text.
        """
        return self._descr.toPlainText()

    def setUserDescription(self, txt):
        """
        Sets the user description.

        Arguments:
            txt (str): User description text.
        """
        self._descr.setPlainText(txt)

    def folder(self):
        """
        Gets the history folder.

        Returns:
            str: Folder path string
        """
        return self._folder.path()

    def setFolder(self, path):
        """
        Sets the history folder.

        Arguments:
            path (str): Folder path string
        """
        self._folder.setPath(path)

    def remoteFolder(self):
        """
        Gets the history folder.

        Returns:
            str: Folder path string
        """
        return self._remote_folder.path()

    def setRemoteFolder(self, path):
        """
        Sets the history folder.

        Arguments:
            path (str): Folder path string
        """
        self._remote_folder.setPath(path)

    def isFolderEnabled(self):
        """
        Gets the history folder enabled state.

        Returns:
            bool: Enabled state
        """
        return self._folder.isEnabled()

    def setFolderEnabled(self, on):
        """
        Sets the history folder enabled state.

        Arguments:
            on (bool): Enabled state
        """
        return self._folder.setEnabled(on)

    def setRemoteFolderEnabled(self, on):
        """
        Sets the history folder enabled state.

        Arguments:
            on (bool): Enabled state
        """
        return self._remote_folder.setEnabled(on)

    def setHistoryVersion(self, version):
        """Sets the version name used in history."""
        self._history_version = version

    def serverPartition(self):
        """
        Gets the server partition
        """
        return self._srvpart.text()

    def setServerPartition(self, srv):
        """
        Sets the server partition
        """
        self._srvpart.setText(srv)

    def qos(self):
        """
        Gets the QOS value
        """
        return self._qos.text()

    def setQos(self, qos):
        """
        Sets the QOS value
        """
        self._qos.setText(qos)

    def runParameters(self):
        """
        Gets all run dialog parameters.
        Do not export MPI parameters for a sequential version.

        Returns:
            dict: Dictionary with all parameters.
        """
        params = {'casename': self.caseName(),
                  'memory': self.memory(), 'time': self.time(),
                  'server': self.server(), 'version': self.codeAsterVersion(),
                  'execmode': self.execMode(),
                  'mode': self.mode(), 'threads': self.threads(),
                  'description': self.userDescription(),
                  'folder': self.folder(),
                  'remote_folder':self.remoteFolder(),
                  'partition': self.serverPartition(),
                  'queue': self.qos()}
        if version_ismpi(self.codeAsterVersion()):
            params.update({'mpicpu' : self.mpiCpu(), 'nodes': self.nodes()})
        return params

    def initRunParameters(self):
        """
        Sets default run dialog parameters.

        Arguments:
            params (dict): Dictionary with run parameters.
        """
        self.setRunParameters(RunPanel.getCachedParams())

    def setRunParameters(self, params):
        """
        Sets run dialog parameters.

        Arguments:
            params (dict): Dictionary with run parameters.
        """
        if params is not None:
            if 'casename' in params:
                self.setCaseName(params['casename'])
            if 'memory' in params:
                self.setMemory(params['memory'])
            if 'time' in params:
                self.setTime(params['time'])
            if 'server' in params:
                self.setServer(params['server'])
            if 'version' in params:
                self.setCodeAsterVersion(params['version'])
            if 'execmode' in params:
                self.setExecMode(params['execmode'])
            if 'mode' in params:
                self.setMode(params['mode'])
            if 'mpicpu' in params:
                self.setMpiCpu(params['mpicpu'])
            if 'threads' in params:
                self.setThreads(params['threads'])
            if 'nodes' in params:
                self.setNodes(params['nodes'])
            if 'description' in params:
                self.setUserDescription(params['description'])
            if 'folder' in params:
                self.setFolder(params['folder'])
            if 'remote_folder' in params:
                self.setRemoteFolder(params['remote_folder'])
            if 'partition' in params:
                self.setServerPartition(params['partition'])
            if 'qos' in params:
                self.setQos(params['qos'])

            self.parametersChanged.emit()

    def isLocal(self):
        """
        Returns *True* if the selected server is the local machine.
        """
        return is_localhost(self.server())

    def isValid(self):
        """
        Returns the run parameters validity state
        """
        hasfolder = self.folder() is not None and len(self.folder()) > 0
        hasrfolder = self.remoteFolder() is not None \
                     and len(self.remoteFolder()) > 0
        needsremote = self._remote_folder.state()
        isvalid = hasfolder and (hasrfolder or not needsremote)
        # ensure a version is selected (in case of error when refeshing server)
        return isvalid and self.codeAsterVersion() != ''

    def _updateState(self):
        """
        Updates the internal controls state depending on assigned case
        """
        case = self.case()
        curcase = self._astergui.study().history.current_case \
            if self._astergui.study() is not None else None

        if case is not None:
            if case == curcase:
                self.initRunParameters()
                self.setReadOnly(False)
            else:
                self.setRunParameters(self._runCaseParameters(case))
                self.setReadOnly(True)
        self.setEnabled(case is not None)

    #pragma pylint: disable=no-self-use
    def _runCaseParameters(self, case):
        """
        Gets the run parameters of given run case.
        """
        params = None
        if case is not None and len(case.stages) > 0:
            params = case.stages[0].result.job.asdict()
            params['casename'] = case.name
        return params

    def _generateName(self):
        """
        Generates the new run case name

        Returns:
            str: String with probably run case name
        """
        return self._astergui.study().history.new_run_case_name()

    def _serverUpdate(self):
        """
        Invoked when server update button clicked
        """
        self._updateServerInfo(True)

    def _serverActivated(self):
        """
        Invoked when current server changed
        """
        self._updateServerInfo(False)

    def _updateServerInfo(self, force=False):
        """
        Updates the code aster versions list dependant from current server.
        """
        wait_cursor(True)
        self._infos.refresh_once(self.server(), force)
        wait_cursor(False)
        self._version.clear()
        versions = self._infos.server_versions(self.server())
        self._version.addItems(versions)
        if self._history_version in versions:
            self.setCodeAsterVersion(self._history_version)
        self._mode.clear()
        self._mode.addItems(self._infos.server_modes(self.server()))
        self.setRemoteFolderEnabled(not self.isLocal())
        # update buttons
        self.parametersChanged.emit()

    def _versionActivated(self):
        """
        Enables/disables the MPI input fields dependant from current
        code_aster version (fields are enabled when version tagged by '_mpi').
        """
        ismpi = version_ismpi(self.codeAsterVersion())
        self._nodes.setEnabled(ismpi)
        self._mpicpu.setEnabled(ismpi)
        params = RunPanel.getCachedParams()
        self.setNodes(params.get('nodes', 1))
        self.setMpiCpu(params.get('mpicpu', 1))
        # execution modes may depend on the version later
        self._execmode.clear()
        self._execmode.addItems(self._infos.exec_modes())

    def _mpicpuEdited(self):
        """Store current as value in cache for future usage."""
        params = RunPanel.getCachedParams()
        params['mpicpu'] = self.mpiCpu()

    def _nodesEdited(self):
        """Store current as value in cache for future usage."""
        params = RunPanel.getCachedParams()
        params['nodes'] = self.nodes()

    def _addWidget(self, grid, label, edit, extra=None):
        """
        Added labeled control into the last grid row.
        """
        row = grid.rowCount()
        grid.addWidget(Q.QLabel(label, self), row, 0)
        grid.addWidget(edit, row, 1)
        if extra is not None:
            grid.addWidget(extra, row, 2)

    def _lastStage(self):
        dsbrd = self._astergui.workSpace().view(Context.Dashboard) \
            if self._astergui.workSpace() is not None else None
        return dsbrd.lastStage() if dsbrd is not None else None
