# -*- coding: utf-8 -*-

# Copyright 2016-2018 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
OpenTurns widget
----------------

This module implements `Export to OpenTurns` panel.

"""

from __future__ import unicode_literals

import os

from PyQt5 import Qt as Q

from . import Entity
from ..common import connect, enable_autocopy, load_pixmap, translate
from ..datamodel.engine import default_parameters
from ..datamodel.parametric import output_commands
from .controller import WidgetController
from .editionwidget import EditionWidget
from .openturns import open_openturns

__all__ = ["OTExportPanel"]

# note: the following pragma is added to prevent pylint complaining
#       about functions that follow Qt naming conventions;
#       it should go after all global functions
# pragma pylint: disable=invalid-name


class OTExportPanel(EditionWidget, WidgetController):
    """OpenTuns Export panel."""

    def __init__(self, astergui, parent=None, **kwargs):
        """
        Create panel.

        Arguments:
            astergui (AsterGui): Parent *AsterGui* instance.
            parent (Optional[QWidget]): Parent widget. Defaults to
                *None*.
            kwargs: Keyword arguments.
        """
        super(OTExportPanel, self).__init__(
            parent=parent,
            name=translate("OTExportPanel",
                           "Export to OpenTurns"),
            astergui=astergui, **kwargs)

        self.setWindowTitle(self.controllerName())
        self.setPixmap(load_pixmap("as_pic_export_to_openturns.png"))

        grid = Q.QGridLayout(self)
        grid.setContentsMargins(0, 0, 0, 0)
        grid.setSpacing(5)

        grid.addWidget(Q.QLabel(translate("OTExportPanel", "Variables"),
                                self), 0, 0)
        self._vars = Q.QListWidget(self)
        self._vars.setObjectName("variables")
        grid.addWidget(self._vars, 0, 1, 2, 1)
        grid.setRowStretch(1, 5)
        no_vars = translate("OTExportPanel", "No Python variable found. "
                            "You must add one or more Python variables "
                            "in a graphical stage of your study.")
        self._vars_empty = Q.QLabel(no_vars, self)
        self._vars_empty.setFrameStyle(Q.QLabel.Box | Q.QLabel.Plain)
        self._vars_empty.setWordWrap(True)
        grid.addWidget(self._vars_empty, 2, 0, 1, 2)
        grid.setRowMinimumHeight(3, 20)

        grid.addWidget(Q.QLabel(translate("OTExportPanel", "Output numpy file"),
                                self), 4, 0)
        self._cmd = Q.QComboBox(self)
        self._cmd.setObjectName("output_cmd")
        grid.addWidget(self._cmd, 4, 1)
        no_cmd = translate("OTExportPanel", "No output command found. "
                           "You must add a <i>IMPR_TABLE</i> command "
                           "using <i>FORMAT='NUMPY'</i> and <i>NOM_PARA</i> "
                           "to define an explicit list of output parameters.")
        self._cmd_empty = Q.QLabel(no_cmd, self)
        self._cmd_empty.setFrameStyle(Q.QLabel.Box | Q.QLabel.Plain)
        self._cmd_empty.setWordWrap(True)
        grid.addWidget(self._cmd_empty, 5, 0, 1, 2)

        grid.setRowStretch(6, 10)

        self._params_empty = Q.QLabel("", self)
        self._params_empty.setWordWrap(True)
        grid.addWidget(self._params_empty, 7, 0, 1, 2)

        connect(self._vars.itemChanged, self.updateButtonStatus)
        connect(self._cmd.currentIndexChanged, self.updateButtonStatus)

        self._initialize()
        self.updateButtonStatus()

    def requiredButtons(self):
        """
        Redefined from *EditionWidget* class.
        """
        return Q.QDialogButtonBox.Ok | Q.QDialogButtonBox.Cancel

    def isButtonEnabled(self, button):
        """
        Redefined from *EditionWidget* class.
        """
        result = True
        if button in (Q.QDialogButtonBox.Ok, Q.QDialogButtonBox.Apply):
            result = len(self.variables()) > 0 and \
                     self.outputCommand() is not None
        return result

    def applyChanges(self):
        """
        Redefined from *EditionWidget* class.
        """
        with enable_autocopy(self.astergui().study().activeCase):
            ot_vars = self.variables()
            ot_cmd = self.outputCommand()
            self.astergui().study().activeCase.set_ot_data(ot_vars, ot_cmd)
            self.astergui().update()
            self.astergui().study().commit(self.controllerName())

    def postClose(self, button):
        """
        Redefined from *EditionWidget* class.
        """
        if button in (Q.QDialogButtonBox.Ok,):
            case = self.astergui().study().activeCase
            ot_data = case.ot_data
            variables = ot_data[0]
            entity = Entity(ot_data[1], 'Command')
            command = self.astergui().study().node(entity)
            open_openturns(case, variables, command)

    def variables(self):
        """Get selected variables."""
        names = []
        for i in range(self._vars.count()):
            item = self._vars.item(i)
            if item.checkState() == Q.Qt.Checked:
                names.append(str(item.text()))
        return names

    def outputCommand(self):
        """Get selected output command."""
        cmd = None
        if self._cmd.count() > 0:
            cmd = self._cmd.currentData()
        return cmd

    def _initialize(self):
        case = self.astergui().study().activeCase
        ot_data = case.ot_data

        variables = sorted(case.variables.keys())
        for variable in variables:
            is_used = variable in ot_data[0]
            item = Q.QListWidgetItem(variable)
            item.setCheckState(Q.Qt.Checked if is_used else Q.Qt.Unchecked)
            self._vars.addItem(item)
        self._vars_empty.setVisible(len(variables) == 0)

        self._cmd.setModel(Model(self.astergui().study().activeCase))
        self._cmd.model().update()
        self._cmd.setCurrentIndex(self._cmd.findData(ot_data[1]))
        self._cmd_empty.setVisible(self._cmd.count() == 0)
        self._update_params_info(case)

    def _update_params_info(self, case):
        """Update the information text about execution parameters."""
        job = case.model.last_params()
        if not job:
            params = default_parameters()
            no_par = (translate("OTExportPanel",
                                "The case has to be executed at least once to "
                                "use the same parameters.<br/>"
                                "No execution found. Default values for "
                                "execution parameters "
                                "({memory} MB, time {time}) will be used.")
                      .format(**params))
        else:
            no_par = (translate("OTExportPanel",
                                "Execution parameters:<br/>"
                                "Server: {server}, memory: {memory} MB, "
                                "time: {time}, version: {version}")
                      .format(**job.asdict()))
        self._params_empty.setText(no_par)


class Model(Q.QStandardItemModel):
    """
    Custom combo-box model for OpenTurns widget.
    """

    def __init__(self, case, parent=None):
        """
        Create model.

        Arguments:
            case (Case): Case object.
            parent (Optional[QObject]): Parent object. Defaults to
                *None*.
        """
        super(Model, self).__init__(parent)
        self._case = case
        self.update()

    def update(self):
        """Update model content."""
        self.clear()

        for command in output_commands(self._case):
            uid = command.uid
            unit = command.storage.get('UNITE')
            stage = command.stage
            filename = stage.handle2file(unit)
            if unit is None or filename is None:
                continue

            item = Q.QStandardItem()
            self.invisibleRootItem().appendRow(item)
            item.setText(os.path.basename(filename))
            item.setData(uid, Q.Qt.UserRole)
            item.setToolTip('{stage} : {name}'.format(stage=stage.name,
                                                      name=filename))
