# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Interface with OpenTurns
------------------------

This module implements the links with the OpenTurns module.

"""

from __future__ import print_function, unicode_literals

import os.path as osp

from ..common import CFG, debug_message, to_str
from ..common.template import BasicFormatter
from ..datamodel.parametric import export_to_openturns

HAS_OTGUI = False
try:
    import otguibase
    HAS_OTGUI = True
except ImportError:
    print("Can not import OpenTurns. Please check your installation.")


def create_yacs_schema(name, code, invars, invalues, indescr, outvars, xml):
    """Create the YACS schema.

    *This function is a workaround waiting for a fix in py2yacs implementation.*

    Arguments:
        name (str): Schema name.
        code (str): Python code of the function.
        invars (list[str]): List of names of input variables.
        invalues (list[float]): List of nominal values of input variables.
        outvars (list[str]): List of names of output variables.
        xml (str): Path to the XML file to create.
    """
    with open(osp.join(CFG.rcdir, "yacs_schema.xml.template"), "rb") as tmpl:
        template = tmpl.read()

    call = "{outvars} = _exec({invars})".format(invars=", ".join(invars),
                                                outvars=", ".join(outvars))
    engine = BasicFormatter()
    text = engine.format(template,
                         schema=name, code=code, call=call,
                         invars=zip(invars, invalues, indescr),
                         outvars=outvars)
    with open(xml, "wb") as output:
        output.write(text)

def create_yacs_model(case, code, invars, invalues, indescr,
                      output_vars):
    """Create the YACS physical model.

    Arguments:
        case (*Case*): Case object.
        code (str): Code of the Python function.
        invars (list[str]): Names of the input variables.
        invalues (list[float]): Nominal values of inputs.
        indescr (list[str]): Descriptions of inputs.
        output_vars (list[str]): Names of output variables.

    Returns:
        *YACSPhysicalModel*: Newly created model.
    """
    yacsxml = osp.join(case.folder, case.name + "_yacs.xml")
    create_yacs_schema(case.name, code, invars, invalues, indescr,
                       output_vars, yacsxml)

    # model = otguibase.YACSPhysicalModel(to_str(case.name), to_str(yacsxml))
    # return model

def create_python_model(case, code, invars, invalues, indescr,
                        output_vars):
    """Create the Python physical model.

    Arguments:
        case (*Case*): Case object.
        code (str): Code of the Python function.
        invars (list[str]): Names of the input variables.
        invalues (list[float]): Nominal values of inputs.
        indescr (list[str]): Descriptions of inputs.
        output_vars (list[str]): Names of output variables.

    Returns:
        *PythonPhysicalModel*: Newly created model.
    """
    input_coll = []
    for name, value, cmt in zip(invars, invalues, indescr):
        var0 = otguibase.Input(to_str(name), value, to_str(cmt))
        input_coll.append(var0)

    output_coll = []
    for out in output_vars:
        res0 = otguibase.Output(to_str(out), to_str(""))
        res0.setValue(0.)
        output_coll.append(res0)

    model = otguibase.PythonPhysicalModel(
        to_str(case.name), input_coll, output_coll, to_str(code))
    return model

def create_otstudy(case, invars, command, with_yacs=False):
    """Create the OpenTurns study.

    Arguments:
        case (Case): Case object.
        invars (list[str]): List of variables.
        command (Command): Output command object.

    Returns:
        OTStudy: OpenTurns study.
    """
    code = export_to_openturns(case, invars, command)
    debug_message("Case name: {0}".format(case.name))
    debug_message("Python function for OpenTurns:\n{0}".format(code))

    study_name = otguibase.OTStudy.GetAvailableName()
    otstudy = otguibase.OTStudy(study_name)
    otguibase.OTStudy.Add(otstudy)

    lvar = [var for name, var in case.variables.items() if name in invars]
    assert len(invars) == len(lvar), (invars, lvar)
    invalues = [var.update() for var in lvar]
    indescr = [var.comment or "" for var in lvar]

    output_vars = command["NOM_PARA"].value

    if with_yacs:
        pass # when create_yacs_model returns a model!
    else:
        create_yacs_model(case, code, invars, invalues, indescr, output_vars)
        model = create_python_model(case, code, invars, invalues,
                                    indescr, output_vars)

    otstudy.add(model)

    return otstudy

def open_openturns(case, invars, command):
    """Open OpenTurns module.

    Arguments:
        case (Case): Case object.
        invars (list[str]): List of variables.
        command (Command): Output command object.
    """
    from .salomegui import get_salome_pyqt
    from .meshview import get_salome_gui

    if not HAS_OTGUI:
        return None

    otstudy = create_otstudy(case, invars, command)
    xmlfile = osp.join(case.folder, otstudy.getName() + ".xml")
    otstudy.save(to_str(xmlfile))
    print("OpenTurns study file is '{0}'".format(xmlfile))

    print("Loading OpenTurns module (please be patient the first time)...")
    otm = get_salome_gui().getComponentUserName(str('OPENTURNS'))
    get_salome_pyqt().activateModule(str(otm))

    studies = otguibase.OTStudy.GetInstances()
    if otstudy not in studies:
        otguibase.OTStudy.Open(to_str(xmlfile))
