# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Mesh view
---------

Implementation of mesh view for SALOME ASTERSTUDY module.

"""

from __future__ import unicode_literals

from PyQt5 import Qt as Q

from ...common import (MeshGroupType, change_cursor, debug_message,
                       is_reference, to_str)
from ..behavior import behavior
from .baseview import MeshBaseView


# note: the following pragma is added to prevent pylint complaining
#       about functions that follow Qt naming conventions;
#       it should go after all global functions
# pragma pylint: disable=invalid-name


def get_smesh_gui():
    """
    Get SMESH GUI Python interface.
    """
    if not hasattr(get_smesh_gui, "smesh_gui"):
        import salome
        get_smesh_gui.smesh_gui = salome.ImportComponentGUI(str('SMESH'))
    return get_smesh_gui.smesh_gui


def find_mesh_by_name(meshfile=None, meshname=None):
    """
    Search mesh object in the SALOME study.

    If *meshname* is not given, last published mesh object from
    *meshfile* is returned.

    If *meshfile* is not given, last published in the study with given
    *meshname* is returned.

    If both *meshfile* and *meshname* are not given, last published
    in the study mesh object is returned.

    Arguments:
        meshfile (Optional[str]): Mesh file name. Defaults to *None*.
        meshname (Optional[str]): Name of the mesh. Defaults to *None*.

    Returns:
        SObject: SALOME study object (*None* if mesh is not found).
    """
    import salome
    import SMESH

    meshobjs = []

    if is_reference(meshfile): # 'meshfile' is entry
        return salome.myStudy.FindObjectID(str(meshfile))

    # find SMESH component
    smesh_component = salome.myStudy.FindComponent(str('SMESH'))
    if smesh_component is not None:
        # iterate through all children of SMESH component, i.e. mesh objects
        iterator = salome.myStudy.NewChildIterator(smesh_component)
        while iterator.More():
            sobject = iterator.Value() # SALOME study object (SObject)
            name = sobject.GetName() # study name of the object
            comment = sobject.GetComment() # file name (see register_meshfile)
            tag = sobject.Tag() # tag (row number)

            # name is empty if object is removed from study
            # tag for mesh object is >= SMESH.Tag_FirstMeshRoot
            if name and tag >= SMESH.Tag_FirstMeshRoot:
                if meshfile is None or comment == to_str(meshfile):
                    if meshname is None or name == to_str(meshname):
                        meshobjs.append(sobject)
            iterator.Next()

    # return last found object (study object)
    return meshobjs[-1] if len(meshobjs) > 0 else None


def register_meshfile(meshes, meshfile):
    """
    Register mesh object.

    The method puts a file name as a 'Comment' attribute to all study
    objects created by importing meshes from that file.

    Arguments:
        meshes (list[Mesh]): SMESH Mesh objects.
        meshfile (str): Mesh file name.
    """
    import salome
    for mesh in meshes:
        sobject = salome.ObjectToSObject(mesh.mesh) # pragma pylint: disable=no-member
        if sobject is not None:
            salome_study = sobject.GetStudy()
            builder = salome_study.NewBuilder()
            attr = builder.FindOrCreateAttribute(sobject,
                                                 str('AttributeComment'))
            attr.SetValue(to_str(meshfile))


def get_salome_gui():
    """
    Get python interface module for SALOME GUI.

    Note:
        Properly deals with `Connect` feature of Salome.
    """
    import salome
    # Deal with `Connect` feature
    if not salome.sg.__dict__:
        import libSALOME_Swig
        return libSALOME_Swig.SALOMEGUI_Swig()
    return salome.sg

class MeshView(MeshBaseView):
    """Central view to display mesh and groups."""

    VIEW_TITLE = "VTK Viewer for Asterstudy"

    def __init__(self, parent=None):
        """
        Create panel.

        Arguments:
            parent (Optional[QWidget]): Parent widget. Defaults to None.

        Note:
            The VTK detached view is not created here but later, on purpose.
            For now, the place where a detached view is put shall
            not be visible at the time when it is created.
            Therefore, the detached view is only created once the
            creation of Asterstudy's desktop is complete.
        """
        MeshBaseView.__init__(self, parent)

        # attached VTK viewer
        self._vtk_viewer = -1

        # define dictionnary to collect displayed object
        self._displayed_entry = dict()
        self._filename2entry = dict()
        self._displayed_mesh = (None, None, 1.0)
        self._displayed_groups = dict()
        self._displayed_normals = dict()

        # supported features
        sm_gui = get_smesh_gui()
        self._features.append('switchable')
        if hasattr(sm_gui, 'setOrientationShown'):
            self._features.append('normals')

        # define current view parameters to avoid unnecessary update
        self._entry2opacity = dict()

    def activate(self):
        """
        Redefined from *MeshBaseView*.

        Create or activate VTK detached view.

        Note:
            The creation of the VTK detached view is not in the initializer.
            The detached view has to be created only once the
            creation of Asterstudy's desktop is complete.
        """
        from .. salomegui import get_salome_pyqt
        sgPyQt = get_salome_pyqt()

        if self._vtk_viewer < 0:
            self._vtk_viewer = sgPyQt.createView(str('VTKViewer'),
                                                 True, 0, 0, True)
            sgPyQt.setViewTitle(self._vtk_viewer, str(MeshView.VIEW_TITLE))

        # put widget within the layout
        view = sgPyQt.getViewWidget(self._vtk_viewer)
        self.layout().addWidget(view)
        view.setVisible(behavior().show_mesh_view)

    def setViewVisible(self, is_on):
        """
        Change mesh view visibility.

        Arguments:
            is_on (bool): Visibility state.
        """
        from .. salomegui import get_salome_pyqt
        sgPyQt = get_salome_pyqt()

        behavior().show_mesh_view = is_on

        if self._vtk_viewer > 0:
            view = sgPyQt.getViewWidget(self._vtk_viewer)
            if view is not None:
                view.setVisible(is_on)

        self._redisplay()

    @staticmethod
    def smesh_types(group_type):
        """
        Get element types for given `group_type`.

        Arguments:
            group_type (int): Group type (*MeshGroupType*).

        Returns:
            list[int]: List of SMESH element types (SMESH type)
        """
        import SMESH
        smesh_types = {
            MeshGroupType.GNode: (SMESH.NODE,),
            MeshGroupType.GElement: (SMESH.EDGE, SMESH.FACE, SMESH.VOLUME,
                                     SMESH.ELEM0D, SMESH.BALL,)
            }
        return smesh_types.get(group_type, [])

    def getMeshEntry(self, meshfile, meshname):
        """
        Get the entry of a mesh in the study.

        Arguments:
            meshfile (str): MED file to read.
            meshname (Optional[str]): Name of the mesh to read.
                If empty, use the first mesh. Defaults to *None*.

        Returns:
            str: entry of the mesh.
        """
        # pragma pylint: disable=no-name-in-module,import-error
        # from this, we create an object in SMESH module
        import salome
        from salome.smesh import smeshBuilder
        smesh = smeshBuilder.New(salome.myStudy)

        if is_reference(meshfile): # 'meshfile' is entry
            return meshfile

        # we get the entry of the mesh in SMESH
        entry = None
        if meshfile not in self._filename2entry:

            # issue27744: test objects already in SMESH first
            # Only create new objects in SMESH if not found
            found = find_mesh_by_name(meshfile, meshname)
            if found is None:
                theMeshes, _ = smesh.CreateMeshesFromMED(to_str(meshfile))
                register_meshfile(theMeshes, meshfile)
                sobject = find_mesh_by_name(meshfile, meshname)
            else:
                sobject = found

            # Register the object to the view
            if sobject is not None:
                entry = sobject.GetID()
                self._filename2entry[meshfile] = {meshname: entry}
        elif meshname not in self._filename2entry[meshfile]:
            sobject = find_mesh_by_name(meshfile, meshname)
            if sobject is not None:
                entry = sobject.GetID()
                self._filename2entry[meshfile][meshname] = entry
        else:
            entry = self._filename2entry[meshfile][meshname]
        return entry

    def alreadyOnDisplay(self, meshfile, meshname):
        """
        Tells if a mesh is already the current mesh on display.

        Arguments:
            meshfile (str): name of the mesh file (a MED format).
            meshname (str): name of mesh within file.

        Returns *True* if it corresponds to the currently displayed mesh.
        """
        return (meshfile, meshname) == self._displayed_mesh[:2]

    @Q.pyqtSlot(str, str, float, bool)
    @change_cursor
    def displayMEDFileName(self, meshfile, meshname=None,
                           opacity=1.0, erase=False):
        """Redefined from *MeshBaseView*."""
        debug_message("entering displayMEDFileName...")

        if not behavior().show_mesh_view:
            self._displayed_mesh = (meshfile, meshname, opacity)
            if erase:
                self._displayed_groups.clear()
                self._displayed_normals.clear()
                self._entry2opacity.clear()
            return

        entry = self.getMeshEntry(meshfile, meshname)

        # activate Asterstudy's VTK view with help of the SalomePyQt utility of
        # SALOME's GUI module
        from .. salomegui import get_salome_pyqt
        get_salome_pyqt().activateViewManagerAndView(self._vtk_viewer)

        # if the mesh is already on display and the view need not be reset
        if self.alreadyOnDisplay(meshfile, meshname) and not erase:

            # if opacity setting differs from the previous one
            if self._entry2opacity.get(entry, None) != opacity:
                self.setAspect(entry, opacity)
                self._displayed_mesh = (meshfile, meshname, opacity)
                get_salome_gui().UpdateView()
            debug_message("displayMEDFileName return #1")
            return

        # display the entry in the active view with help of the `sg` python
        # module of `salome` python package

        self._displayed_entry.clear()
        self._displayed_groups.clear()
        self._displayed_normals.clear()
        self._entry2opacity.clear()
        get_salome_gui().EraseAll()
        get_salome_gui().Display(str(entry))

        self._displayed_mesh = (meshfile, meshname, opacity)
        self._displayed_entry[entry] = 1
        self.setAspect(entry, opacity)
        get_salome_gui().FitAll()
        get_salome_gui().UpdateView()
        debug_message("displayMEDFileName return final")

    @Q.pyqtSlot(str, str, str, int)
    @change_cursor
    def displayMeshGroup(self, meshfile, meshname, group, grtype):
        """Redefined from *MeshBaseView*."""
        if grtype not in self._displayed_groups:
            self._displayed_groups[grtype] = dict()
        self._displayed_groups[grtype][group] = True
        if not behavior().show_mesh_view:
            return

        import SMESH

        sm_gui = get_smesh_gui()

        # import MED file and register meshes if needed
        self.getMeshEntry(meshfile, meshname)

        sobject = self.find_group_by_name(meshfile, meshname, group, grtype)
        if sobject is None:
            return

        entry = sobject.GetID()

        # activate Asterstudy's VTK view with help of the SalomePyQt utility of
        # SALOME's GUI module
        from .. salomegui import get_salome_pyqt
        get_salome_pyqt().activateViewManagerAndView(self._vtk_viewer)

        # go for display
        get_salome_gui().Display(str(entry))
        self._displayed_entry[entry] = 1

        if behavior().grp_global_cmd:
            if self.isFeatureSupported('normals'):
                tag = sobject.GetFather().Tag()
                if tag in (SMESH.Tag_FaceGroups, SMESH.Tag_VolumeGroups):
                    # pragma pylint: disable=no-member
                    sm_gui.setOrientationShown(str(entry),
                                               self.allNormalsShown(),
                                               self._vtk_viewer)

        self.setAspect(entry, opacity=1.0)
        get_salome_gui().UpdateView()

    @Q.pyqtSlot(str, str, str, int)
    def undisplayMeshGroup(self, meshfile, meshname, group, grtype):
        """Redefined from *MeshBaseView*."""
        if grtype not in self._displayed_groups:
            self._displayed_groups[grtype] = dict()
        self._displayed_groups[grtype][group] = False
        if not behavior().show_mesh_view:
            return

        sobject = self.find_group_by_name(meshfile, meshname, group, grtype)
        if sobject is None:
            return

        entry = sobject.GetID()

        if self._displayed_entry.get(entry):
            # activate Asterstudy's VTK view with help of the SalomePyQt
            # utility of SALOME's GUI module
            from .. salomegui import get_salome_pyqt
            get_salome_pyqt().activateViewManagerAndView(self._vtk_viewer)

            # go for display
            get_salome_gui().Erase(str(entry))
            self._displayed_entry[entry] = 0

            get_salome_gui().UpdateView()

    def setAspect(self, entry, opacity):
        """Set aspect of an object."""

        # to set up display options
        # let us retrieve an instance of libSMESH_Swig.SMESH_Swig
        sm_gui = get_smesh_gui()
        if not hasattr(sm_gui, 'GetActorAspect'):
            # Customizing aspect requires a newer GUI module.
            return

        # pragma pylint: disable=no-member

        # retrieve a structure with presentation parameters for this actor
        pres = sm_gui.GetActorAspect(str(entry), self._vtk_viewer)
        # redefine some of these parameters
        pres.opacity = opacity
        self._entry2opacity[entry] = opacity

        # In this method, the opacity was changed
        # The color is another parameter that can be changed, with:
        # >>> pres.surfaceColor.r = 0.
        # >>> pres.surfaceColor.g = 1.
        # >>> pres.surfaceColor.b = 0.
        #
        # This possibility is not yet used in Asterstudy but could be soon.
        #
        # Note: the above code has to be adapted according to dimension
        # volumeColor  : 3D
        # surfaceColor : 2D
        # edgeColor    : 1D
        # nodeColor    : 0D
        #

        # reinject this for the actor
        sm_gui.SetActorAspect(pres, str(entry), self._vtk_viewer)

    def find_group_by_name(self, meshfile, meshname, groupname,
                           grtype=MeshGroupType.GElement):
        """
        Search mesh group.

        Arguments:
            meshfile (str): Mesh file name.
            meshname (str): Name of the mesh.
            groupname (str): Name of the group.
            grtype (int): Type of the group (nodes or elements).

        Returns:
            SObject: SALOME study object (*None* if group is not found).
        """
        # TODO: replace all this with a call to SMESH service GetGroupByName
        sobject = find_mesh_by_name(meshfile, meshname)
        if sobject is not None:
            salome_study = sobject.GetStudy()
            import SMESH
            # iterate through all types of groups
            for tag in range(SMESH.Tag_FirstGroup, SMESH.Tag_LastGroup+1):
                ok, container = sobject.FindSubObject(tag)
                if not ok or container is None:
                    continue
                # iterate through all group of particular type
                iterator = salome_study.NewChildIterator(container)
                while iterator.More():
                    child = iterator.Value()
                    name = child.GetName()
                    typ = child.GetObject().GetType()
                    if name and name == to_str(groupname):
                        if typ in self.smesh_types(grtype):
                            return child
                    iterator.Next()
        return None

    def normalsShown(self, meshfile, meshname, group):
        """Redefined from *MeshBaseView*."""
        import SMESH

        sm_gui = get_smesh_gui()
        if not self.isFeatureSupported('normals'):
            return None

        if not behavior().show_mesh_view:
            return self._displayed_normals.get(group)

        sobject = self.find_group_by_name(meshfile, meshname, group)
        if sobject is None:
            return None

        entry = sobject.GetID()
        tag = sobject.GetFather().Tag()

        if tag not in (SMESH.Tag_FaceGroups, SMESH.Tag_VolumeGroups):
            return None

        if self._displayed_entry.get(entry):
            # pragma pylint: disable=no-member
            return sm_gui.isOrientationShown(str(entry), self._vtk_viewer)

        return None

    def showNormals(self, meshfile, meshname, group, visible):
        """Redefined from *MeshBaseView*."""
        self._displayed_normals[group] = visible
        if not behavior().show_mesh_view:
            return

        sm_gui = get_smesh_gui()
        if not self.isFeatureSupported('normals'):
            return

        if behavior().grp_global_cmd:
            return

        sobject = self.find_group_by_name(meshfile, meshname, group)
        if sobject is None:
            return

        entry = sobject.GetID()
        if self._displayed_entry.get(entry):
            # pragma pylint: disable=no-member
            sm_gui.setOrientationShown(str(entry), visible, self._vtk_viewer)

    def _updateNormalsVisibility(self):
        """Redefined from *MeshBaseView*."""
        sm_gui = get_smesh_gui()
        if not self.isFeatureSupported('normals'):
            return

        if behavior().grp_global_cmd:
            import salome
            import SMESH
            for dentry in self._displayed_entry:
                sobject = salome.myStudy.FindObjectID(str(dentry))
                if sobject is None:
                    continue
                tag = sobject.GetFather().Tag()
                if tag not in (SMESH.Tag_FaceGroups, SMESH.Tag_VolumeGroups):
                    continue
                # pragma pylint: disable=no-member
                sm_gui.setOrientationShown(str(dentry), self.allNormalsShown(),
                                           self._vtk_viewer)

    def _redisplay(self):
        """Redisplay all mesh data."""
        all_groups = dict.copy(self._displayed_groups)
        normals = dict.copy(self._displayed_normals)
        meshfile, meshname, opacity = self._displayed_mesh

        self._displayed_mesh = (None, None, 1.0)

        if meshfile is None or meshname is None:
            return

        self.displayMEDFileName(meshfile, meshname, opacity)
        for grtype, groups in all_groups.iteritems():
            for group, visible in groups.iteritems():
                if visible:
                    self.displayMeshGroup(meshfile, meshname, group, grtype)
                else:
                    self.undisplayMeshGroup(meshfile, meshname, group, grtype)
        for group, visible in normals.iteritems():
            self.showNormals(meshfile, meshname, group, visible)
        self._updateNormalsVisibility()
