# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Test utilities for GUI
----------------------

Auxiliary functions and classes used for testing of AsterStudy
application.

"""

from __future__ import unicode_literals

from PyQt5 import Qt as Q

from . import Context, NodeType, check_selection
from ..common import get_file_name
from ..datamodel import Validity

__all__ = ["DebugWidget", "GuiTester"]

# note: the following pragma is added to prevent pylint complaining
#       about functions that follow Qt naming conventions;
#       it should go after all global functions
# pragma pylint: disable=invalid-name


class GuiTester(object):
    """Test actions for GUI."""

    DefaultStudy = 1
    DumpStudy = 2
    TestDialog = 3

    def __init__(self, astergui):
        """
        Constructor. Create test actions.

        Arguments:
            astergui (AsterGui): AsterGui instance.
        """
        self._astergui = astergui
        self._actions = {}
        #
        action = Q.QAction("Create sample study", astergui)
        action.setStatusTip("Create a sample study for testing")
        action.triggered.connect(self._create_sample_study)
        self._actions[self.DefaultStudy] = action
        #
        action = Q.QAction("Dump study", astergui)
        action.setStatusTip("Dump study contents to the file")
        action.triggered.connect(self._dump_study)
        self._actions[self.DumpStudy] = action
        self._dump_file_name = None
        #
        action = Q.QAction("Test dialog", astergui)
        action.setStatusTip("Test dialog")
        action.triggered.connect(self._test_dialog)
        self._actions[self.TestDialog] = action

    @property
    def astergui(self):
        """Get AsterGui instance."""
        return self._astergui

    def actions(self):
        """Get test actions."""
        return self._actions.values()

    def updateActions(self):
        """Update actions state."""
        has_study = self.astergui.study() is not None
        is_empty_study = not has_study or \
            self.astergui.study().history.current_case.nb_stages == 0
        self._actions[self.DefaultStudy].setEnabled(is_empty_study)
        self._actions[self.DumpStudy].setEnabled(has_study)
        self._actions[self.TestDialog].setEnabled(has_study)

    def _create_sample_study(self):
        """Action slot: Create sample study."""
        if self.astergui.study() is None:
            self.astergui.newStudy()
        case = self.astergui.study().history.current_case
        if case.nb_stages > 0:
            return
        stage_1 = case.create_stage()
        stage_1.add_command('DEFI_MATERIAU', 'material_1')
        stage_1.add_command('DEFI_MATERIAU', 'material_2')
        stage_1.add_command('AFFE_MODELE', 'model')
        stage_1.add_command('DEFI_MATERIAU', 'material_3')
        case.create_stage()
        self.astergui.study().commit("Sample study creation")
        self.astergui.update()

    def _dump_study(self):
        """Action slot: Dump study contents to file."""
        if self.astergui.study() is None:
            return
        if self._dump_file_name is None:
            file_name = get_file_name(0, self.astergui.mainWindow(),
                                      "Dump Study", "", "*.txt")
            if file_name:
                self._dump_file_name = file_name
        if self._dump_file_name is not None:
            try:
                dump_file = open(self._dump_file_name, "w")
                dump_file.write(dump_history(self.astergui.study().history))
                dump_file.close()
            except IOError:
                pass

    def _test_dialog(self):
        """Action slot: Test some dialog."""
        # Put a code to test your dialog below.
        # For example:
        # dialog = MyDialog(self.astergui.mainWindow())
        # dialog.resize(600, 200)
        # dialog.exec_()


class DebugWidget(Q.QLineEdit):
    """
    Helper widget for advanced testing with Squish.

    To use it:

    #. Call `setText(property)`: specify property to check for currently
       selected object.
    #. Call `text()` to get value of the property.
    """

    def __init__(self, astergui):
        """
        Create widget.

        Arguments:
            astergui (AsterGui): Parent AsterGui instance.
        """
        super(DebugWidget, self).__init__(astergui.mainWindow())
        self.setObjectName("DebugWidget")
        self.astergui = astergui
        self.setReadOnly(True)
        self.textChanged.connect(self.process)
        self.setContextMenuPolicy(Q.Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self._contextMenu)

        self.props = {
            "validity" : self._validity,
            "edit mode" : self._editMode,
            "cmd for doc" : self._command4Doc,
            "has favorites" : self._hasFavorites,
            }

    def process(self, text):
        """
        Called when `setText()` is called (programmatically).

        Arguments:
            text (str): Property being checked.
        """
        self.blockSignals(True)
        func = self.props.get(text)
        if func:
            func()
        self.blockSignals(False)

    def _validity(self):
        selected = self.astergui.selected(Context.DataSettings)
        if check_selection(selected, size=1):
            node = self.astergui.study().node(selected[0])
            validity = node.check()
            text = "valid"
            if validity != Validity.Nothing:
                text = "invalid: "
                what = []
                if validity & Validity.Syntaxic:
                    what.append("syntaxic")
                if validity & Validity.Dependency:
                    what.append("dependency")
                if validity & Validity.Naming:
                    what.append("naming")
                text += ", ".join(what)
            self.setText(text)

    def _editMode(self):
        selected = self.astergui.selected(Context.DataSettings)
        if check_selection(selected, size=1, typeid=NodeType.Stage):
            stage = self.astergui.study().node(selected[0])
            self.setText("graphical" if stage.is_graphical_mode() else "text")

    def _command4Doc(self):
        cmd = self.astergui._commandForDoc() # pragma pylint: disable=protected-access
        self.setText(cmd.title if cmd else "nothing")

    def _hasFavorites(self):
        favoritesMgr = self.astergui.favoritesMgr()
        self.setText("true" if favoritesMgr.commands() else "false")

    def _contextMenu(self, pos):
        menu = Q.QMenu(self)
        for prop in self.props:
            menu.addAction(prop)
        action = menu.exec_(self.mapToGlobal(pos))
        if action is not None:
            self.setText(action.text())


def dump_history(history):
    """
    Dump model to a string representation.

    Arguments:
        history (History): Data model.

    Returns:
        str: String representation of whole data model.
    """
    result = []

    nodes = (history.get_node(uid) for uid in history.uids)
    for node in nodes:
        result.append(repr(node))

    return '\n'.join(result) + "\n"
