# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Favorites manager
-----------------

Implementation of favorite commands manager for AsterStudy application.

"""

from __future__ import unicode_literals

import os

from PyQt5 import Qt as Q

from ..common import (CFG, get_absolute_dirname, read_file, translate,
                      write_file)
from ..datamodel import CATA

# note: the following pragma is added to prevent pylint complaining
#       about functions that follow Qt naming conventions;
#       it should go after all global functions
# pragma pylint: disable=invalid-name


class FavoritesMgr(object):
    """Favorites manager."""

    def __init__(self):
        """Create manager."""
        self._commands = []

        self._dirName = get_absolute_dirname(CFG.userrc)
        self._fileName = os.path.join(self._dirName, "asterstudy_favorites")

    @staticmethod
    def name():
        """
        Get name of the manager (uses as text of the corresponding menu action
        and title of the category in `Show all` panel).

        Returns:
            str: Name of the manager.
        """
        return translate("AsterStudy", "Favorites")

    def commands(self):
        """
        Get list of favorite commands (excluding the commands that do not
        belong to the current catalog.

        Returns:
            list[str]: List of commands.
        """
        commands = []
        for command in self._commands:
            if CATA.get_catalog(command):
                commands.append(command)
        return commands

    def hasCommand(self, command):
        """
        Check that the specified command is added to favorites.

        Arguments:
            command (str): Command to be checked.

        Returns:
            bool: *True* if the command is added to favorites; *False*
            otherwise.
        """
        return True if command in self._commands else False

    def addCommand(self, command):
        """
        Add the specified command to favorites.

        Arguments:
            command (str): Command to be added.
        """
        if not self.hasCommand(command):
            self._commands.append(command)
        self.save()

    def removeCommand(self, command):
        """
        Remove the specified command from favorites.

        Arguments:
            command (str): Command to be removed.
        """
        if self.hasCommand(command):
            self._commands.remove(command)
        self.save()

    def load(self):
        """ Load the favorite commands from the resource file. """
        self._commands = []

        if not os.path.exists(self._fileName):
            return self._commands

        content = read_file(self._fileName)
        if content:
            self._commands = content.splitlines()

    def save(self):
        """ Save the favorite commands to the resource file. """
        content = "\n".join(self._commands)

        try:
            if not os.path.isdir(self._dirName):
                os.makedirs(self._dirName)
            write_file(self._fileName, content)
        except IOError:
            message = translate("AsterStudy", "Cannot write file.")
            Q.QMessageBox.critical(self, "AsterStudy", message)
