# -*- coding: utf-8 -*-

# Copyright 2016-2018 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Open With action
----------------

The module implements action that manages *Open With* functionality.

"""

from __future__ import unicode_literals

import os
import os.path as osp
from ConfigParser import Error, SafeConfigParser

from PyQt5 import Qt as Q

from ...common import CFG, get_absolute_dirname, get_extension, translate

__all__ = ["OpenWithAction"]

# note: the following pragma is added to prevent pylint complaining
#       about functions that follow Qt naming conventions;
#       it should go after all global functions
# pragma pylint: disable=invalid-name


class OpenWithAction(Q.QAction):
    """Action that manages Open With functionality."""

    activated = Q.pyqtSignal(str)
    """
    Signal: emitted when action is triggered.

    Arguments:
        executable (str): Selected program.
    """

    def __init__(self, parent):
        """
        Create action.

        Arguments:
            parent (Optional[QObject]): Parent object. Defaults to *None*.
        """
        super(OpenWithAction, self).__init__(parent)
        self.setText(translate("OpenWithAction", "Open With"))
        self.setMenu(Q.QMenu())
        self.menu().aboutToShow.connect(self._updateMenu)

        self._file_name = None

        self._mapper = Q.QSignalMapper(self)
        self._mapper.mapped[str].connect(self._useProgram)

        userrc_dir = get_absolute_dirname(CFG.userrc)
        app_file = osp.join(userrc_dir, "asterstudy_apps")
        self._mgr = Q.QSettings(app_file, Q.QSettings.IniFormat)

    def setFileName(self, file_name):
        """
        Set file name to action.

        Arguments:
            file_name (str): File path.
        """
        self._file_name = file_name

    @Q.pyqtSlot()
    def _updateMenu(self):
        """Fill in menu."""
        self.menu().clear()
        if self._file_name and osp.exists(self._file_name):
            ext = get_extension(self._file_name)
            if not ext:
                ext = '__without_extension__'
            known_exts = self._mgr.childGroups()
            if ext in known_exts:
                self._mgr.beginGroup(ext)
                keys = sorted(self._mgr.childKeys())
                for key in keys:
                    program = self._mgr.value(key)
                    if not osp.exists(program):
                        continue
                    icon = Q.QIcon(find_icon(program))
                    action = self.menu().addAction(icon, osp.basename(program))
                    self._mapper.setMapping(action, program)
                    action.triggered.connect(self._mapper.map)
                if self.menu().actions():
                    self.menu().setDefaultAction(self.menu().actions()[0])
                self._mgr.endGroup()
            self.menu().addSeparator()
            text = translate("OpenWithAction", "Choose Program...")
            action = self.menu().addAction(text)
            action.triggered.connect(self._chooseProgram)

    @Q.pyqtSlot(bool)
    def _chooseProgram(self):
        """Choose program."""
        title = translate("OpenWithAction", "Choose program")
        program, _ = Q.QFileDialog.getOpenFileName(self.parent().mainWindow(),
                                                   title)
        if program:
            self._useProgram(program)

    @Q.pyqtSlot(str)
    def _useProgram(self, program):
        """Called when program has been chosen from menu."""
        if not program:
            return
        ext = get_extension(self._file_name)
        if not ext:
            ext = '__without_extension__'
        self._mgr.beginGroup(ext)
        keys = sorted(self._mgr.childKeys())
        programs = []
        for key in keys:
            programs.append(self._mgr.value(key))
            self._mgr.remove(key)
        if program in programs:
            programs.remove(program)
        programs[0:0] = [program]
        for i, j in enumerate(programs):
            self._mgr.setValue('app{:03d}'.format(i+1), j)
        self._mgr.endGroup()
        self.activated.emit(program)


def find_icon(app, size=32):
    """
    Find icon for given application.

    Arguments:
        app (str): Path to application's executable.
        size (Optional[int,str]): Preferable icon size.
            Defaults to 32x32.
    Returns:
        str: Path to the icon file or *None* if icon is not found.
    """
    if app:
        dir_path = osp.dirname(app)
        app_name = osp.basename(app)
        home_dir = os.getenv('HOME', '')
        desktop_file = '{}.desktop'.format(app_name)
        desktop_files = []
        desktop_files.append(osp.join(dir_path, desktop_file))
        desktop_files.append(osp.join(home_dir, 'Desktop', desktop_file))
        cfg = SafeConfigParser()
        cfg.read(desktop_files)
        try:
            return cfg.get('Desktop Entry', 'Icon')
        except Error:
            pass
        resolutions = ['hicolor', 'locolor']
        sizes = [32, 36, 42, 48, 24, 22, 16, 64, 72, 96, 8, 128, 192, 256]
        if size in sizes:
            sizes.remove(size)
        if size:
            sizes[0:0] = [size]
        couples = [(i, j) for i in resolutions for j in sizes]
        for i, j in couples:
            size_dir = '{0}x{0}'.format(j)
            icon_dir = osp.join('/usr/share/icons', i, size_dir, 'apps')
            for ext in ('.png', '.jpg', '.jpeg', '.xpm'):
                icon_file = osp.join(icon_dir, app_name + ext)
                if osp.exists(icon_file):
                    return icon_file
    return None
