# -*- coding: utf-8 -*-

# Copyright 2016-2018 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
List action
-----------

The module implements action which is represented as a submenu in menus
and as a combo-box in toolbars.

"""

from __future__ import unicode_literals

from PyQt5 import Qt as Q

from ...common import update_visibility
from ..widgets import ShrinkingComboBox

__all__ = ["ListAction"]

# note: the following pragma is added to prevent pylint complaining
#       about functions that follow Qt naming conventions;
#       it should go after all global functions
# pragma pylint: disable=invalid-name


class ListAction(Q.QWidgetAction):
    """Action that shows list of child items."""

    triggered = Q.pyqtSignal(str)
    """
    Signal: emitted when child action is triggered.

    Arguments:
        text (str): Child item's text.
    """

    highlighted = Q.pyqtSignal(str)
    """
    Signal: emitted when child action is highlighted.

    Arguments:
        text (str): Child item's text.
    """

    def __init__(self, text, icon=None, parent=None):
        """
        Create list action with given menu `text` and, optionaly,
        `icon`.

        Arguments:
            text (str): Action's menu text.
            icon (Optional[QIcon]): Action's icon. Defaults to *None*.
            parent (Optional[QObject]): Parent object. Defaults to *None*.
        """
        super(ListAction, self).__init__(parent)
        self.setText(text)
        if icon is not None:
            self.setIcon(icon)
        self.setMenu(Q.QMenu())
        self._mapper = Q.QSignalMapper(self)
        self._mapper.mapped[str].connect(self.triggered[str])
        self.changed.connect(self._changed)

    def addItem(self, text, ident=None, subcategory=None):
        """
        Add item into the action.

        Arguments:
            text (str): Item's text.
            ident (Optional[str]): Item's identifier.

        Returns:
            QAction: Action that corresponds to just added item.
        """
        menu = None
        if subcategory:
            for item in self.menu().actions():
                if item.menu() is not None and item.text() == subcategory:
                    menu = item.menu()
            if menu is None:
                items = [i for i in self.menu().actions() if i.menu() is None]
                before = items[0] if items else None
                menu = Q.QMenu(subcategory, self.menu())
                self.menu().insertMenu(before, menu)
        else:
            menu = self.menu()
        action = menu.addAction(text)
        action.setData(ident)
        self._mapper.setMapping(action, ident if ident is not None else text)
        action.triggered.connect(self._mapper.map)
        self._changed()
        return action

    def clear(self):
        """
        Clear list action.
        """
        self.menu().clear()
        self._changed()

    def count(self):
        """
        Get number of items in the action.

        Returns:
            int: Number of items.
        """
        return len(self.menu().actions())

    def createWidget(self, parent_widget):
        """
        Create widget for action.

        Arguments:
            parent_widget (QWidget): Parent widget.

        Returns:
            QWidget: Created widget.
        """
        if isinstance(parent_widget, (Q.QMenu, Q.QMenuBar)):
            # For menu: show sub-menu.
            widget = super(ListAction, self).createWidget(parent_widget)
        else:
            # For toolbar: show drop-down combo box.
            widget = ShrinkingComboBox(parent_widget)
            widget.setFocusPolicy(Q.Qt.NoFocus)
            self._updateWidget(widget)
            widget.activated.connect(self._itemActivated)
            widget.highlighted.connect(self._itemHighlighted)
            widget.setEnabled(self.isEnabled())
            widget.setToolTip(self.toolTip())
            widget.setStatusTip(self.statusTip())
        return widget

    def _updateWidget(self, widget):
        """
        Fill the widget with the list of child items.

        Arguments:
            widget (QComboBox): Associated combo box.
        """
        widget.clear()
        widget.addItem(self.icon(), self.text())
        actions = self.menu().actions()

        def _addItem(_action):
            idx = widget.count()
            widget.addItem(_action.text(), _action.data())
            widget.setItemData(idx, _action.toolTip(), Q.Qt.ToolTipRole)
            widget.setItemData(idx, _action.statusTip(), Q.Qt.StatusTipRole)

        def _addSeparator(_action):
            widget.addSeparator(_action.text())

        items = [i for i in actions if i.menu() is None]
        menus = [i for i in actions if i.menu() is not None]
        for action in items:
            _addItem(action)
        for menu in menus:
            _addSeparator(menu)
            for action in menu.menu().actions():
                _addItem(action)

    @Q.pyqtSlot(int)
    def _itemActivated(self, idx):
        """
        Called when combo box's item is activated.

        Emits `triggered(str)` signal.

        Arguments:
            idx (int): Item's index.
        """
        control = self.sender()
        control.blockSignals(True)
        control.setCurrentIndex(0)
        control.blockSignals(False)
        if idx > 0 and not control.isSeparator(idx):
            data = control.itemData(idx)
            text = control.itemText(idx)
            self.triggered.emit(data if data is not None else text)

    @Q.pyqtSlot(int)
    def _itemHighlighted(self, idx):
        """
        Called when combo box's item is highlighted.

        Emits `highlighted(str)` signal.

        Arguments:
            idx (int): Item's index.
        """
        control = self.sender()
        if idx > 0 and not control.isSeparator(idx):
            data = control.itemData(idx)
            text = control.itemText(idx)
            self.highlighted.emit(data if data is not None else text)

    @Q.pyqtSlot()
    def _changed(self):
        """
        Called when action is changed.

        Updates related widgets.
        """
        for widget in self.createdWidgets():
            widget.setEnabled(self.isEnabled())
            widget.setItemIcon(0, self.icon())
            widget.setItemText(0, self.text())
            widget.setToolTip(self.toolTip())
            widget.setStatusTip(self.statusTip())
            self._updateWidget(widget)
        for widget in self.associatedWidgets():
            update_visibility(widget)
