# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Parametric study
----------------

Implementation of the functions to deal with parametric studies.
"""

from __future__ import unicode_literals
from __future__ import print_function

import os.path as osp

from .engine.engine_utils import ExportCase

INPUTS = 'inputs.pick'
OUTPUTS = 'outputs.npy'


def output_commands(case):
    """Search for commands that create output results for parametric studies.

    Arguments:
        case (Case): Case to search in.

    Returns:
        list[Command]: List of commands that match the criteria.
    """
    def _check_cmd(_cmd):
        _unit = _cmd.storage.get('UNITE')
        return (_cmd.title == 'IMPR_TABLE' and
                _unit is not None and
                _cmd.stage.handle2file(_unit) is not None and
                _cmd.storage.get('FORMAT') == 'NUMPY' and
                len(_cmd.storage.get(('NOM_PARA'), ())) > 0)
    found = []
    for stage in case.stages:
        found.extend([i for i in stage.sorted_commands if _check_cmd(i)])
    return found

def create_exec_function(casedir, input_vars, output_vars, verbose=1):
    """Create the 'exec' for parametric study as expected by OpenTurns.

    Arguments:
        casedir (str): Path to the parametric case directory (must contain the
            related export file named 'parametric.export').
        input_vars (list[str]): List of the names of the variable parameters.
        output_vars (list[str]): List of the names of the results.
        verbose (int): Verbosity level (default: 1, verbosy). 0 means silent.

    Returns:
        str: Code of the function.
    """
    template = \
"""
def _exec({input_vars}):
    from asterstudy.api import ParametricCalculation

    calc = ParametricCalculation("{casedir}",
                                 [{input_names}], [{input_vars}])
    calc.set_verbosity({verbose})
    calc.run()
    results = calc.output_values()
    if len(results) != {len_output_vars}:
        raise ValueError("Calculation failed with inputs: {{0}}"
                         .format(({input_vars})))

    {output_vars} = results{first}
    return {output_vars}
"""
    values = {
        'casedir': casedir,
        'input_vars': ", ".join(input_vars),
        'input_names': ", ".join(["'{0}'".format(i) for i in input_vars]),
        'output_vars': ", ".join(output_vars),
        "len_output_vars": len(output_vars),
        'first': "" if len(output_vars) > 1 else "[0]",
        'verbose': verbose,
    }
    code = template.format(**values)
    with open(osp.join(casedir, 'parametric.code'), 'wb') as fcode:
        fcode.write(code)
    return code

def export_to_openturns(case, input_vars, output_command):
    """Import the parametric study in OpenTurns.

    Arguments:
        case (Case): Case to search in.
        input_vars (list[str]): List of the names of the variable
            parameters.
        output_command (Command): Command object that write the results.

    Return:
        str: Code of the function + open the OpenTurns module.
    """
    output_vars = output_command['NOM_PARA'].value
    # get Info object created by 'output_command'
    stage = output_command.stage
    info = None
    for i in stage.handle2info.viewvalues():
        if output_command in i:
            info = i
            break
    # set the expected filename
    bckp = info.filename
    info.filename = OUTPUTS

    export_name = osp.join(case.folder, "parametric.export")
    export = ExportCase.factory(case, export_name, parametric=True)
    export.set_last_stage(stage.number)
    export.generate()

    # restore original path
    info.filename = bckp

    code = create_exec_function(case.folder, input_vars, output_vars)
    return code
