# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
EDF servers adapters
--------------------

Implementation of functions specific to EDF servers.

"""

from __future__ import unicode_literals

from ...common import to_str


def adapt_parameters(salome_job, params):
    """Adapt *JobParameters* depending on server configuration.

    Arguments:
        salome_job (*JobParameters*): SALOME object for the parameters.
        params (dict): Dict of parameters provided by the GUI.
    """
    # compute memory per node in GB to choose partition
    nbnodes = max(1, params.get('nodes', 1))
    cpu_per_node = 1. * params.get('mpicpu', 1) / nbnodes

    memgb = params.get('memory') * cpu_per_node / 1024.

    if params.get('partition'):
        salome_job.partition = to_str(params['partition'])
    else:
        if to_str(params['server']) == 'aster5':
        # partitions sur aster5 : 'cn64' , 'cn256' , 'cn512' , 'comp' , 'bmem'
            if memgb > 256:
                salome_job.partition = to_str('cn512')
            elif memgb > 64:
                salome_job.partition = to_str('cn256')
            else:
                salome_job.partition = to_str('cn64')
        elif to_str(params['server']) == 'eole':
        # noms des partitions sur eole : 'cn' , 'bm'
            if memgb > 115:
                salome_job.partition = to_str('bm')
            else:
                salome_job.partition = to_str('cn')

    if params.get('queue'):
        salome_job.queue = to_str(params['queue'])

    if params.get('mpicpu') > 1:
        salome_job.exclusive = True
