# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Text DataSet
------------

Implementation of *TextDataSet*.

"""

from __future__ import unicode_literals

from ...common import no_new_attributes
from ..command import Command
from ..catalogs import CATA

from .base import DataSet

class PatchedDataSet(DataSet):
    """Implementation of a semi graphical and text dataset.
    """

    __setattr__ = no_new_attributes(object.__setattr__)

    def __init__(self):
        """Create dataset."""
        DataSet.__init__(self)
        self._text = ""

    def __len__(self):
        """No definition for this *hybrid* dataset."""
        raise NotImplementedError('must be subclassed')

    def add_command(self, command_type, name):
        """Add a command into the dataset.

        Arguments:
            command_type (str): Type of the command (in a catalogue).
            name (Optional[str]): Name of the command. Defaults to
                *None*; in this case name is automatically generated for
                the command.

        Returns:
            Command: Command just added.
        """
        cata = CATA.get_catalog(command_type)
        syntax_checker = CATA.package('Syntax').SyntaxCheckerVisitor
        command = Command.factory(name, command_type, cata, syntax_checker)

        self._model.add(command, self)

        return command


class TextDataSet(PatchedDataSet):
    """Implementation of the text dataset.

    Contains only the text of a code_aster commands file.
    """

    _mode = DataSet.textMode

    _text = None
    __setattr__ = no_new_attributes(object.__setattr__)

    def __init__(self):
        """Create dataset."""
        PatchedDataSet.__init__(self)
        self._text = ""

    @property
    def text(self):
        """str: Attribute that holds *text* of the TextDataSet."""
        return self._text

    @text.setter
    def text(self, text):
        self._text = text

    def __len__(self):
        """Get the dataset length: number of lines in the text.

        Returns:
            int: Length of dataset which is equal to the number of lines
            of the text.
        """
        return len(self._text.strip().splitlines())

    def append_text(self, text):
        """Append text to the content.

        Arguments:
            text (str): Text being added.
        """
        self._text += text

    def __mul__(self, other):
        """Support native Python '*' operator protocol."""
        assert self.text == other.text

    def initialize_copy(self):
        """
        Create a copy with no connections.

        Text is copied into content"""

        res = self.__class__()
        res.text = self.text
        return res

    @property
    def sorted_commands(self):
        """Returns the list of commands (as *commands*) because the commands
        are not ordered in a Text dataset.
        """
        return self.commands
