# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D i
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Command Helpers
---------------

Helper functionality for Command implementation.

"""

from __future__ import unicode_literals

from ...common import recursive_items, recursive_setter
from ..abstract_data_model import add_parent, remove_parent
from ..visit_study import FilterVisitor, UnitVisitor
from ..general import ConversionLevel, FileAttr
from ..catalogs import CATA

def register_cos(stage, command, pool_co=None):
    """Properly register all CO command args via producing corresponding
    hiddens.

    Arguments:
        stage (Stage): The stage that the command belongs to.
        command (Command): Command object.
        pool_co (dict, optional): CO objects that existed before unregistering.

    Returns:
        dict: Pool of CO objetcs that have not been reused.
    """
    for path, value in command.get_list_co():
        exist = pool_co and pool_co.get(path)
        if not exist:
            hidden = stage.add_command("_RESULT_OF_MACRO", value.name)
            hidden.init(dict(DECL=value, PARENT=command, PATH=path))
        else:
            del pool_co[path]
            exist.name = value.name
            add_parent(exist, command)

def unregister_cos(command, delete=True):
    """Unregister all registered command COs via removing corresponding hiddens.

    .. note:: If the caller does not need the CO objects, it must delete them.

    Arguments:
        command (Command): Command object that have to forget CO objects.
        delete (bool): If *True*, CO objects that belongs to the command are
            deleted. Otherwise, the command is only removed from their parents
            list.

    Returns:
        dict: Pool of CO objetcs that have not been forgotten.
    """
    pool_co = {}
    for child in command.hidden:
        if child.parent_id == command.uid:
            # command is the creator of the CO
            if delete:
                child.delete()
            else:
                pool_co[child.storage_nocopy['PATH']] = child
                # The Hidden must be removed if not re-registered.
                remove_parent(child, command)
        else:
            remove_parent(child, command)
    return pool_co

def unregister_parent(command, value):
    "Unregister all value items from the command instance"
    from .basic import Command

    if isinstance(value, Command):
        remove_parent(command, value)
    elif isinstance(value, (list, tuple)):
        for item in value:
            unregister_parent(command, item)
    elif isinstance(value, dict):
        for item in value.itervalues():
            unregister_parent(command, item)

def update_dependence_up(storage):
    """Update 'up' dependencies of a command.

    When copying a command, this ensures that it will
    be referencing copies not originals up the dependence tree.

    This command is useful when parents are copied
        before children.

    Arguments:
        storage (dict): storage object with the values of the
            current command/keyword."""
    #
    # The method is called when copying children
    #
    # Before the method is called
    # -------   ------------
    # | copy |  | original |  parent commands
    # -------   -----------
    #          ^         ^
    #         /          |    reference
    #        /           |
    # -------   ------------
    # | copy |  | original |  child commands
    # -------   -----------
    #
    # After the method has been called
    # -------   ------------
    # | copy |  | original |  parent commands
    # -------   -----------
    #     ^              ^
    #     |              |    reference
    #     |              |
    # -------   ------------
    # | copy |  | original |  child commands
    # -------   -----------
    #
    from .basic import Command
    for path, item in recursive_items(storage):
        if isinstance(item, Command) and item.ccopy is not None:
            recursive_setter(storage, path, item.ccopy)

def update_dependence_down(command, same_stage):
    """Update 'down' dependencies of a command.

    When copying a command, this ensures that other
    copied commands down the dependence tree
    will be referencing the copy not the original.

    This method is useful when children are copied
        before parents.

    Arguments:
         command(Command): command object
         same_stage(bool): *True* updates only dependence
             between commands in the same stage,
             *False* updates only dependence
             between commands in different stages.
         """
    #
    # The method is called when copying parents
    #
    # Before the method is called
    # -------   ------------
    # | copy |  | original |  parent commands
    # -------   -----------
    #          ^         ^
    #         /          |    reference
    #        /           |
    # -------   ------------
    # | copy |  | original |  child commands
    # -------   -----------
    #
    # After the method has been called
    # -------   ------------
    # | copy |  | original |  parent commands
    # -------   -----------
    #     ^              ^
    #     |              |    reference
    #     |              |
    # -------   ------------
    # | copy |  | original |  child commands
    # -------   -----------
    #
    if command.ccopy is not None:
        if same_stage:
            iterator = (i.ccopy for i in command.child_commands \
                       if i.ccopy is not None \
                           and i.ccopy.stage.number == command.stage.number)
            for child in iterator:
                _update_dependence_helper(child, command)
        if not same_stage:
            iterator = (i for i in command.child_commands \
                        if i.stage.number > command.stage.number and \
                           i.stage.parent_case is not command.stage.parent_case)
            for child in iterator:
                _update_dependence_helper(child, command)

def _update_dependence_helper(command, parent):
    def _predicate(simple):
        return simple.value is parent

    visitor = FilterVisitor(_predicate)
    command.accept(visitor)

    for keyword in visitor.keywords:
        keyword.value = parent.ccopy

def register_parent(command, value):
    "Register value items in the command instance"
    from .basic import Command

    if isinstance(value, Command):
        if value.name == '_' and value.gettype() is not None:
            value.name = "_%d" % value.uid
        add_parent(command, value)
    elif isinstance(value, (list, tuple)):
        for item in value:
            register_parent(command, item)
    elif isinstance(value, dict):
        for item in value.itervalues():
            register_parent(command, item)


def clean_undefined(value):
    "Removes all 'keys' of None 'values'"
    if isinstance(value, (list, tuple)):
        for item in value:
            clean_undefined(item)
    elif isinstance(value, dict):
        for key, item in value.items():
            if item is None:
                del value[key]
            else:
                clean_undefined(item)


def unregister_unit(command, decrement=True, clear=True):
    """
    Unregister all units assigned to the command.

    Arguments:
        decrement (bool): decrement reference count
            of *Info* object to *Command* object ?
        clear (bool): delete *Info* object if no longer
            used in any command ?
    """
    # Track unused units
    stage = command.stage
    if clear:
        stage.track_unused()

    # Get all UNITE keywords
    visitor = UnitVisitor()
    try:
        command.accept(visitor) # the command is probably invalid
    except KeyError:
        return

    if not visitor.keywords:
        return

    for keyword in visitor.keywords:

        # Value of UNITE keyword (see Simple.value)
        # (as stored in the `storage` dictionnary of command content)
        value = keyword.value

        # TODO: common subfunction with `register_unit` to check validity
        # no need to test *None* value here,
        # will have been discarded by visitor
        if isinstance(value, dict):
            if len(value) != 1:
                continue
            unit = value.keys()[0]
            if unit is None:
                continue
        if isinstance(value, int):
            unit = value

        info = stage.handle2info.get(unit)
        if info is not None:
            # Decrement reference count
            if decrement:
                if command not in info:
                    continue
                del info[command]

            # Delete info if reference count has dropped to zero
            if clear and not len(info):
                del stage.handle2info[unit]


def register_unit(command):
    """
    Find out and register all units assigned to the command.

    Note: this function also has the responsability to check
        that the values it provides to UNITE keywords are valid.
    """
    # Get all UNITE keywords
    visitor = UnitVisitor()
    command.accept(visitor)

    stage = command.stage
    for keyword in visitor.keywords:
        value = keyword.value

        # If the provided value is an *int*
        # The corresponding filename is assumed to be given by the dictionary
        if isinstance(value, int):
            info = stage.handle2info[value]
            val = keyword.cata.definition['inout']
            attr = FileAttr.str2value(val)
            info[command] = attr
            continue

        # Put *None* in the event of an empty or invalid *dict*
        if value is None or len(value) != 1:
            keyword.value = None
            continue

        filename = value.values()[0]
        unit = value.keys()[0]

        if unit is None:
            keyword.value = None
            continue

        info = stage.handle2info[unit]
        info.filename = filename if len(filename) > 0 else None

        val = keyword.cata.definition['inout']
        attr = FileAttr.str2value(val)

        info[command] = attr
        keyword.value = unit

def register_deleter(command, just_created):
    """Only the DETRUIRE commands that already exist before *command*
    are considered at creation.
    For other modifications add deleters from preceding stages only.

    Arguments:
        command (*Command*): Command in which the parent will be added.
        just_created (bool): If *True*, *command* is just being created.
    """
    deleters = command.preceding_deleters(only_preceding=not just_created)
    for cmd in deleters:
        register_parent(command, cmd)

def unregister_deleter(deleter):
    """Unregister commands that are children of a *deleter* command (called
    before the deleter is deleted).

    Arguments:
        deleter (*Command*): The deleter that is unparented.
    """
    from .basic import Command
    if not deleter.is_deleter:
        return

    for cmd in deleter.child_nodes:
        if isinstance(cmd, Command):
            remove_parent(cmd, deleter)
            cmd.reset_validity()

def avail_meshes(storage):
    """Mesh concepts of whom the commands in storage depend."""
    from .basic import Command
    res = []
    for val in storage.viewvalues():
        if isinstance(val, dict):
            res += avail_meshes(val)
        elif isinstance(val, (list, tuple)):
            for elem in val:
                if isinstance(elem, dict):
                    res += avail_meshes(elem)
                elif isinstance(elem, Command):
                    res += avail_meshes_in_cmd(elem)
        elif isinstance(val, Command):
            res += avail_meshes_in_cmd(val)

    # eliminate duplicates in list
    return list(set(res))

def avail_meshes_in_cmd(command):
    """Mesh concepts of whom the command depends."""
    from .basic import Command
    res = []
    # `command` may be None if the user did not specify any command
    if command is None:
        return res
    typ = command.gettype(ConversionLevel.NoFail)
    if typ is CATA.package("DataStructure").maillage_sdaster:
        res.append(command)
    parents = [cmd for cmd in command.parent_nodes \
                           if isinstance(cmd, Command)]
    for par in parents:
        res += avail_meshes_in_cmd(par)

    # eliminate duplicates in list
    return list(set(res))

def deleted_by(command):
    """Return the list of results deleted by a DETRUIRE Command.

    Arguments:
        command (Command): Command to analyze.

    Returns:
        list[Command]: List of Command
    """
    from .mixing import Sequence
    if not command.is_deleter:
        return []

    deleted = []
    lfact = command["CONCEPT"]
    is_seq = isinstance(lfact, Sequence)
    if not is_seq:
        lfact = [lfact]
    for fact in lfact:
        obj = fact["NOM"].value
        obj = obj if isinstance(obj, (list, tuple)) else [obj]
        deleted.extend(obj)
    # cleanup else CONCEPT is set
    if is_seq and lfact.undefined(): # pragma pylint: disable=no-member
        del command._engine["CONCEPT"] # pragma pylint: disable=protected-access

    return deleted
