# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D i
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Basic Command
-------------

Implementation of the basic Command.

"""

from __future__ import unicode_literals

import copy

from ...common import no_new_attributes, recursive_items, to_str
from ..abstract_data_model import Node, add_parent
from ..catalogs import CATA
from ..general import CataMixing, ConversionLevel
from ..general import CopyContext as Ctx
from ..general import Validity
from ..visit_study import FilterVisitor
from .constancy import ModifiesCommandInstance as ModifiesInstance
from .helper import (clean_undefined, deleted_by, register_cos,
                     register_deleter, register_parent, register_unit,
                     unregister_cos, unregister_deleter, unregister_parent,
                     unregister_unit, update_dependence_up)
from .mixing import CO, KeysMixing, ResultMixing


class Command(Node, CataMixing, ResultMixing):
    """Implementation of the command"""
    name_length = 8
    subclass_dict_cache = None
    specific_name = None
    is_co = False
    _cache_type = None
    _validity = _check_validity = None
    _engine = _storage = _syntax_checker = _title = _active = None
    _reuse_input_name = None
    ccopy = None
    __setattr__ = no_new_attributes(object.__setattr__)

    @staticmethod
    def factory(name, title, cata, syntax_checker):
        """Create a generic or specific Command object."""
        if not Command.subclass_dict_cache:
            dict_sub = {}
            for cls in Command.__subclasses__(): # pragma pylint: disable=no-member
                if cls.specific_name:
                    dict_sub[cls.specific_name] = cls
            Command.subclass_dict_cache = dict_sub

        cls = Command.subclass_dict_cache.get(title, Command)

        return cls(name, title, cata, syntax_checker)

    def __init__(self, name, title, cata, syntax_checker):
        """Constructor"""
        Node.__init__(self, name)
        CataMixing.__init__(self, cata)
        self._active = True
        self._engine = None
        self._storage = {}
        self.cata = cata
        self._syntax_checker = syntax_checker
        self._title = title
        self.ignore_copy = Command
        self.ignore_parent = Command

        self._validity = Validity.Nothing
        self._check_validity = True
        self.ccopy = None
        self.reset_cache_type()

    def __deepcopy__(self, memodict):
        # pragma pylint: disable=unused-argument
        """Supports native Python 'deepcopy' function protocol"""
        return self

    @property
    def active(self):
        """Attribute that holds 'active' status"""
        return self._active

    @active.setter
    def active(self, value):
        self._active = value

        comment = self.comment
        if comment:
            comment.active = value

        if value:
            register_unit(self)
        else:
            unregister_unit(self)

        from .hidden import Hidden
        for concept in self.child_nodes:
            if isinstance(concept, Hidden):
                concept.active = value

        self.reset_validity()

    @property
    def storage(self):
        """Attribute that holds parameter's internal representation"""
        return copy.deepcopy(self._storage)

    @property
    def storage_nocopy(self):
        """Return the parameter's internal representation (for performance).

        Warning: The caller must ensure that the object won't be changed.
        """
        return self._storage

    @CataMixing.cata.setter # pragma pylint: disable=no-member
    def cata(self, value):
        "Declares setter for so named property"
        keywords = value.definition
        CataMixing.cata.fset(self, value) # pragma pylint: disable=no-member
        self._engine = KeysMixing(value, keywords, self._storage, self)

    def rkeys(self):
        """Returns definition *keys* in the composite structure"""
        return self._engine.rkeys()

    def keys(self):
        """Returns actual *keys* in the composite structure"""
        return self._engine.keys()

    def __getitem__(self, name):
        """Returns composite structure item for the given *key*"""
        # self.reset_validity() has been added by 3c3d87ec4168
        # but __getitem__ is a read-only action because KeysMixing.__getitem__
        # works on a copy of 'storage'.
        return self._engine[name]

    @ModifiesInstance(True)
    def __setitem__(self, name, value):
        """Assigns a value for the given *key*"""
        self.reset_validity()
        self._engine[name] = value

    def gettype(self, strict=ConversionLevel.Type):
        """Returns code_aster type for the given command"""
        if strict & ConversionLevel.Type:
            return self.unsafe_type()
        return self.safe_type()

    def unsafe_type(self):
        """Calculates code_aster type; raises exception in case"""
        if self._cache_type:
            return self._cache_type
        typ = self.cata.get_type_sd_prod(**self._storage)
        # if the function returns None but it should not, return `baseds`
        if CATA.expects_result(self.cata) and (not typ or typ == type(None)):
            raise TypeError
        self._cache_type = typ
        return typ

    def safe_type(self):
        """Returns code_aster type for the given command in a safe way"""
        try:
            return self.unsafe_type()
        except: # pragma pylint: disable=bare-except
            return CATA.baseds

    def get_possible_types(self):
        """Returns the list of all possible types that a command may create.

        Returns:
            list[DataStructure] or list[list[DataStructure]]: List of possible
            types or a list of list if there are additional results for
            macro-commands.
        """
        if self._model.support_all_types:
            return self.cata.get_all_types()
        else:
            return [CATA.baseds]

    def submit(self):
        """Submits the command changes into the Study"""
        self.reorder()
        self.reset_validity()
        self._auto_naming()

    def get_list_co(self):
        """Return the list of *CO* found in a storage dict with their path
        in the storage dict."""
        cos = []
        for key, node in recursive_items(self._storage):
            if isinstance(node, CO):
                cos.append((key, node))
        return cos

    @property
    def list_co(self):
        """Return the list of *CO* passed as argument."""
        return [i[1] for i in self.get_list_co()]

    @property
    def hidden(self):
        """Return the list of produced *Hidden* commands."""
        hidden = []
        from .hidden import Hidden
        for node  in self.child_nodes:
            if isinstance(node, Hidden):
                hidden.append(node)
        return hidden

    def _register_cos(self, pool_co):
        """
        Properly register all CO command args via producing corresponding
        hiddens
        """
        register_cos(self.stage, self, pool_co)

    @ModifiesInstance(True)
    def init(self, storage, duplication=False, conversion=False):
        """
        Set command content dictionary to `storage`, discards earlier content

        Initializes its context from an outside dictionary.

        During the duplication process, do not create Hidden objects from CO
        because they will be created by duplication of the previously existing
        ones.

        Arguments:
            storage (dict): dictionnary of command content
            duplication (bool): *True* if this is done in the context
                of a duplication
            conversion (bool): *True* if this is done in the context
                of a comm2study conversion
        """
        pool_co = unregister_cos(self, delete=False)
        unregister_unit(self, decrement=True, clear=False)
        unregister_parent(self, self._storage)

        just_created = len(self._storage) == 0
        self._storage.clear()
        self._engine.clear_cache()

        clean_undefined(storage)
        update_dependence_up(storage)

        self._storage.update(storage)

        register_parent(self, self._storage)
        register_deleter(self, just_created)

        # in case of duplication, do not create new COs
        if not duplication:
            self._register_cos(pool_co)
        for hid in pool_co.itervalues():
            hid.delete()

        if self.active:
            register_unit(self)

        if not conversion:
            unregister_unit(self, decrement=False, clear=True)

        self.submit()

        return self

    def __call__(self, storage):
        """Initializes its context from an outside dictionary"""
        self.init(storage)
        return self

    def __mul__(self, other):
        """Supports native Python '*' operator protocol."""
        assert str(self) == str(other)

    def __eq__(self, other):
        """Supports native Python '==' operator protocol."""
        return self is other

    def __nonzero__(self):
        """Implements truth value testing '*if command*'"""
        # else __len__ would be used
        return True

    def short_repr(self):
        """Returns a short textual representation."""
        return to_str("{0.name} <from {0._title}>".format(self))

    def node_repr(self):
        """Native node representation that shows dependencies."""
        return Node.__repr__(self)

    def __str__(self, rename=False):
        """Stringifies a command: returns is code_aster syntax"""
        from ..study2comm import ExportToCommVisitor
        export = ExportToCommVisitor(rename=rename)

        comment = self.comment # Print corresponding comment first
        if comment is not None:
            comment.accept(export) # pragma pylint: disable=no-member

        self.accept(export)

        return to_str(export.get_text())

    def __repr__(self):
        """Raw representation"""
        return self.node_repr()

    @property
    def stage(self):
        """Returns corresponding parent *Stage* instance"""
        return next(node.stage for node in self.parent_nodes \
                    if not isinstance(node, Command))

    def preceding_deleters(self, only_preceding):
        """Collects all the previously existing deleters.

        Arguments:
            only_preceding (bool): If *True* only returns the deleters from the
                preceding stages, not from the current. If *False* all the
                existing deleters are returned.

        Returns:
            list[Command]: List of deleters.
        """
        preceding_commands = self.stage.preceding_commands(self, only_preceding)
        return [cmd for cmd in preceding_commands if cmd.is_deleter]

    @staticmethod
    def filterby(stage, astype, command=None):
        """Collects all commands with given *astype*.

        Commands that returned in the reversed order of *sorted_commands*.
        """
        excommands = reversed(stage.preceding_commands(command))

        filtered = []
        for item in excommands:
            typ = item.gettype(strict=ConversionLevel.NoFail)

            if typ is None:
                continue

            if isinstance(astype, basestring) \
                    and isinstance(typ, basestring):
                if typ in ['R', 'I'] and astype in ['R', 'I']:
                    filtered.append(item)
                elif typ == astype:
                    filtered.append(item)
            elif not isinstance(astype, basestring) \
                    and not isinstance(typ, basestring) \
                    and issubclass(typ, astype):
                filtered.append(item)

        return filtered

    def groupby(self, astype):
        """Collects all commands in the Aster-Study Case with given *astype*"""
        return Command.filterby(self.stage, astype, self)

    @property
    def title(self):
        """Attribute that holds unique *title*"""
        return self._title

    def need_reuse(self):
        """Tell if the command needs the 'reuse' argument.
        It is required if it the same name of an input keyword is used as the
        result name."""
        for _, node in recursive_items(self._storage):
            if isinstance(node, Command) and node.name == self.name:
                return True
        return False

    def can_reuse(self):
        """Tell if the command can reuse the result name."""
        return self._cata.can_reuse()

    def reused(self):
        """Return the object that is reused if any.

        The rules of the catalog should limit the possibility to only
        one object.

        Returns:
            tuple: A tuple containing a boolean telling if the command must
            modify an object or not, and the object that is or can be reused or
            *None* if it isn't found.
        """
        raw = self._cata.definition.get("reentrant", "").split(':')
        required = raw[0] == 'o'
        if len(raw) < 2:
            return required, None
        obj = None
        raw.pop(0)
        for keyword in raw[0].split("|"):
            obj = self._storage.get(keyword, None)
            if obj is None:
                continue
            if len(raw) > 1:
                iobj = obj
                if not isinstance(obj, (list, tuple)):
                    iobj = [obj, ]
                for obj in iobj:
                    obj = obj.get(raw[1], None)
                    if obj is not None:
                        break
            else:
                break
        return required, obj

    @property
    def reuse_input_name(self):
        """Attribute that tells if the user want to name the result as an
        input objects"""
        return self._reuse_input_name

    @reuse_input_name.setter
    def reuse_input_name(self, value):
        """Set value of the property.

        Arguments:
            value (bool): *True* to reuse the name of an input object if found.
        """
        self._reuse_input_name = value

    def keywords_equal_to(self, value):
        """Return the keywords that are equal to *value*."""
        def _predicate(simple):
            return simple.value == value

        visitor = FilterVisitor(_predicate)
        self.accept(visitor)
        return visitor.keywords

    def accept(self, visitor):
        """Walks along the objects tree using the visitor pattern."""
        visitor.visit_command(self)

    @ModifiesInstance(True)
    def rename(self, name):
        """Renames the result of the command."""
        self.name = name

    @ModifiesInstance(True)
    def copy(self):
        """Create and return copied Command node."""
        return self._model.copy(self, context=Ctx.UserCopy)

    def initialize_copy(self):
        """Creates a copy with no connections"""
        return Command.factory(self._name, self._title,
                               self._cata, self._syntax_checker)

    def finalize_copy(self, orig, **kwargs):
        """
        Hook: attune data model additionally after a Command
        has been copied. `self` is the copy.

        Updates storage of the newly created Command.

        Arguments:
            orig (Command): Original object.
            context (Ctx): Context of duplication.
        """
        context = kwargs.get('context', Ctx.Nothing)

        # tag copies
        if not context & Ctx.UserCopy:
            orig.ccopy = self

        if orig is not None:
            self.active = orig.active
            self.init(orig.storage, duplication=True)

        # update dependence downwards
        if not context & Ctx.UserCopy:
            from .helper import update_dependence_down
            update_dependence_down(orig, True)
            update_dependence_down(orig, False)

    def autocopy(self):
        """Calls parent stage's autocopy method"""
        self.stage.autocopy()

    @property
    def comment(self):
        "Returns related Comment Command instance"
        from .comment import Comment
        for command in self.parent_nodes:
            if isinstance(command, Comment):
                return command

        return None

    @comment.setter
    def comment(self, content):
        "Adds or updates Comment instance for the given command"
        if content is None:
            return

        comment = self.comment
        if comment is None:
            stage = self.stage
            comment = stage.add_comment(content, concatenate=False)
            add_parent(self, comment)
            comment.reorder()
        else:
            comment.content = content

    def delete(self, user_deletion=False):
        """Delete command from the model.

        Arguments:
            user_deletion (bool): *True* if this is an explicit individual
                command deletion from the user, *False* if this is called
                as a consequence of stage deletion.
        """
        if user_deletion:
            self._decorated_delete()
        else:
            self._undecorated_delete()

    @ModifiesInstance(True)
    def _decorated_delete(self):
        "Decorated delete method, trigger auto copy if necessary"
        Node.delete(self)

    @ModifiesInstance(False)
    def _undecorated_delete(self):
        "Undecorated delete method, never triggers autocopy"
        Node.delete(self)

    def before_remove(self):
        """Prepares to remove the command from the model.

        Removes child hidden commands that have been automatically added.
        """
        unregister_cos(self)
        unregister_deleter(self)

        comment = self.comment
        if comment is not None:
            comment.delete() # pragma pylint: disable=no-member

        unregister_unit(self, clear=self.stage.is_graphical_mode())

        self.stage.on_remove_command(self)

        self.reset_validity()

        self.reset_cache_type()

        return Node.before_remove(self)

    def reset_validity(self):
        """Recursively resets cachable validity flag"""
        for child in self.child_nodes:
            child.reset_validity()

        self._check_validity = True

    def reset_cache_type(self):
        """Recursively resets cachable type flag"""
        for child in self.child_nodes:
            child.reset_cache_type()

        self._cache_type = None

    def check(self, mode=Validity.Complete, safe=True):
        """Checks given validity aspect and returns corresponding status

        Arguments:
            mode (Validity): Defines the level of checking (see
                general.Validity for levels definition).
            safe (bool): If `safe` is False, an error is raised in case of
                Syntaxic error.

        Returns:
            Validity: Status of the validation.
        """
        if not self.active:
            return Validity.Nothing

        check_all = mode == Validity.Complete
        if not self._check_validity and check_all:
            return self._validity

        result = Validity.Nothing

        if mode & (Validity.Naming | Validity.Dependency):
            if not self._check_naming():
                result |= Validity.Naming
            if not self._check_naming_conflicts():
                result |= Validity.Naming

        if mode & Validity.Dependency:
            result |= self._check_dependencies()

        if mode & Validity.Syntaxic:
            result |= self._check_syntax(check_all, safe)

        if check_all:
            self._validity = result
            self._check_validity = False

        return result

    def reorder(self):
        """Check the command position in the dataset."""
        if self.stage.is_graphical_mode():
            self.stage.reorder(self)

    def _auto_naming(self):
        """Automatically set the command result name in case of 'reuse'."""
        required, obj = self.reused()
        if (required or self.reuse_input_name) and obj:
            self.name = obj.name

    def _check_naming(self):
        """Check the validity of the command result name."""
        if self._name == "_":
            return True

        if len(self._name) > self.name_length:
            return False

        # is it a valid python variable?
        try:
            exec self._name + " = 0" # pragma pylint: disable=exec-used
        except SyntaxError:
            return False

        return True

    def previous_names(self):
        """Returns command names to be considered for naming conflicts"""
        preceding_commands = self.stage.preceding_commands(self)
        names = set([cmd.name for cmd in preceding_commands if cmd != self])
        return names

    def _check_naming_conflicts(self):
        """Check the validity of the command result name relatively to all
        preceding commands of the case (including commands from preceding
        stages)."""
        if self.name == "_" or self.can_reuse():
            return True

        if self.name in self.previous_names():
            return False

        return True

    def _check_dependencies(self):
        """Checks dependencies."""
        result = Validity.Nothing

        deleters = [cmd for cmd in self.parent_nodes
                    if isinstance(cmd, Command) and cmd.is_deleter]

        for parent in self.parent_nodes:
            if result & Validity.Dependency:
                break
            if parent not in self._model:
                result |= Validity.Dependency
            elif isinstance(parent, Command):
                if parent.active:
                    # do not propagate Naming conflict to children
                    # else all the model must be rechecked to fix it
                    result |= parent.check() & ~Validity.Naming
                    # parent previously deleted?
                    for rmcmd in deleters:
                        if parent in deleted_by(rmcmd):
                            result |= Validity.Dependency
                else:
                    result |= Validity.Dependency

        return result

    def _check_syntax(self, check_all, safe):
        """Checks syntax of the command."""
        result = Validity.Nothing

        checker = self._syntax_checker()
        if not safe:
            self._cata.accept(checker, self.storage_nocopy)
            if check_all:
                self._check_validity = False
            return result

        try:
            self._cata.accept(checker, self.storage_nocopy)
        except Exception: # pragma pylint: disable=broad-except
            result |= Validity.Syntaxic

        return result

    def repair(self, previous_commands):
        """Try to repair the stage in case of dependency error.

        - Search for broken dependencies: commands that are not in the model.

        - Try to fix these broken dependencies by using results with the same
          name and type.

        Returns:
            bool: True if a replacement has be done (but that does not ensure
                that the Case is valid).
        """
        def _search_compatible_commands(lost):
            for command in previous_commands:
                if command.name != lost.name:
                    continue
                if command.gettype() == lost.gettype():
                    return command
            return None

        valid = self.check(mode=Validity.Dependency)
        if valid & Validity.Dependency:
            # search for keywords with broken dependencies
            def _predicate(simple):
                cmd = simple.value
                if not isinstance(cmd, Command):
                    return False
                return cmd not in self._model

            visitor = FilterVisitor(_predicate)
            self.accept(visitor)

            # replace broken deps
            for kwd in visitor.keywords:
                replaceby = _search_compatible_commands(kwd.value)
                if replaceby is not None:
                    kwd.value = replaceby

    @property
    def is_prologue(self):
        """[bool]: True for the prologue commands, False for the rest."""
        return self._title in ("DEBUT", "POURSUITE")

    @property
    def is_deleter(self):
        """[bool]: *True* for the DETRUIRE command, *False* otherwise."""
        return self._title == "DETRUIRE"

    @property
    def categ(self):
        """[int]: Category index of the command in the data tree."""
        if self.is_prologue:
            return -999
        elif self._title in ("FIN",):
            return 999
        elif self._title in ("_CONVERT_VARIABLE",):
            return -1000
        if self._title in ("_CONVERT_COMMENT",):
            children = [i for i in self.child_nodes if isinstance(i, Command)]
            if children:
                return children[0].categ
        return CATA.get_category_index(self._title)

    def _after_rename(self):
        """Called when command is renamed."""
        # Renaming could break validity in case of naming conflict
        self.reset_validity()

    @property
    def child_commands(self):
        """Commands that depend upon this one"""
        # Do NOT put `type(self)` instead of `Command` here
        return [i for i in self.child_nodes if isinstance(i, Command)]
