# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""
Some utilities for remote files.

"""
# This module is only covered by remote testcases
# that's why it is excluded from coverage report.

from __future__ import print_function, unicode_literals

import os
import os.path as osp
import urlparse
import subprocess as SP

from PyQt5 import Qt as Q

from .base_utils import is_localhost, no_new_attributes, to_str

FILE_TRANSFER_PROTOCOL = "sftp"
HOST_SEP = "@"
PATH_SEP = ":"


__test__ = int(os.getenv("ASTERSTUDY_WITHIN_TESTS", "0"))


def is_gnome():
    """Tests whether Desktop Environment is Gnome."""
    if getattr(is_gnome, "cache", None) is None:
        try:
            # pragma pylint: disable=import-error
            from gi.repository import Gio as gio, GLib as glib
        except ImportError:
            gio, glib = None, None
        is_gnome.cache = gio, glib
    return is_gnome.cache


class MountedVolumes(object):
    """Store the volumes mounted during the session, to be able to unmount them
    when the application exits.
    """
    uri = None
    __setattr__ = no_new_attributes(object.__setattr__)

    def __init__(self):
        self.uri = set()

    def exists(self, path):
        """Tests existence of files located on a server.

        Note:
            The file system of the server has to be mounted
            on Gnome virtual filesystem for this function
            to work.

        Arguments:
            path (str): URL of the remote file.

        Returns:
            bool: *True* if the the path exists, *False* otherwise.
        """
        gio = is_gnome()[0]
        if gio is None:
            return False

        previously_mounted = path in self.uri
        self.mount(path)
        fobj = gio.File.new_for_commandline_arg(path)
        exist = bool(fobj.query_exists())
        if previously_mounted:
            self.unmount(path)
        return exist

    def mount(self, path):
        """Mount enclosing filesystem to path with Gio.

        Arguments:
            path (str): URI or path to be mount.
        """
        gio, glib = is_gnome()
        if gio is None:
            return

        # Set up everything to start mount loop
        # new_for_uri or new_for_path? we don't know
        fobj = gio.File.new_for_commandline_arg(path)
        loop = glib.MainLoop()
        mount_operation = gio.MountOperation()
        mount_operation.set_anonymous(True)

        # Define callback
        def _cbck_end(g_file, result):
            """Callback when mount operation is over."""
            try:
                if g_file.mount_enclosing_volume_finish(result):
                    self.uri.add(path)
                    self.print("Filesystem {0} mounted".format(path))
            except glib.GError as err: # pragma pylint: disable=catching-non-exception
                # Pass already mounted case silently
                if err.code == gio.IOErrorEnum.ALREADY_MOUNTED: # pragma pylint: disable=no-member
                    pass
                elif not __test__:
                    self.print("Mounting {0} filesystem: failed".format(path))
                    self.print(err)
            finally:
                # End loop anyway
                loop.quit()

        # Define desired mount
        fobj.mount_enclosing_volume(gio.MountMountFlags.NONE,
                                    mount_operation, None, _cbck_end)
        # Run
        loop.run()

    def unmount(self, path):
        """Unmount an enclosing filesystem.

        Arguments:
            path (str): URI or path to be mount.
        """
        gio, glib = is_gnome()
        if gio is None:
            return

        fobj = gio.File.new_for_commandline_arg(path)
        gobj = fobj.find_enclosing_mount()
        loop = glib.MainLoop()
        mount_operation = gio.MountOperation()
        mount_operation.set_anonymous(True)

        # Define callback
        def _cbck_end(g_mount, result):
            """Callback when mount operation is over."""
            try:
                if g_mount.unmount_with_operation_finish(result):
                    self.print("Filesystem {0} unmounted".format(path))
            except glib.GError as err: # pragma pylint: disable=catching-non-exception
                if not __test__:
                    self.print("Unmounting {0} filesystem: failed".format(path))
                    self.print(err)
            finally:
                # End loop anyway
                loop.quit()

        # Define desired mount
        gobj.unmount_with_operation(gio.MountMountFlags.NONE,
                                    mount_operation, None, _cbck_end)

        # Run
        loop.run()

    def unmount_all(self):
        """Unmount all known mount points."""
        for uri in self.uri:
            self.unmount(uri)

    @staticmethod
    def print(msg):
        """Wrapper on *print*."""
        if __test__:
            return
        print(msg)

# local instance to manage mount points
MOUNT = MountedVolumes()

# static utilities
def exists_remote(path):
    """Tests existence of files located on a server.

    Note:
        The file system of the server has to be mounted
        on Gnome virtual filesystem for this function
        to work.

    Arguments:
        path (str): URL of the remote file.

    Returns:
        bool: *True* if the the path exists, *False* otherwise.
    """
    return MOUNT.exists(path)

def mount_enclosing_fs(path):
    """Mount enclosing filesystem to path with Gio."""
    MOUNT.mount(path)

def unmount_enclosing_fs(path):
    """Unmount enclosing filesystem to path with Gio."""
    MOUNT.unmount(path)

def unmount_all():
    """Unmount all filesystems mounted during this session."""
    MOUNT.unmount_all()


def dlg2url(path):
    """
    Convert path returned by a file dialog into valid url.

    Arguments:
        path (str): file path as returned by a QFileDialog.
    """
    gio = is_gnome()[0]
    if gio is None:
        return path

    vfs = gio.Vfs.get_default()
    gio_fobj = vfs.get_file_for_path(Q.QUrl(path).path())
    fileurl = gio_fobj.get_uri()

    # Return full url if remote, else return unchanged path
    is_remote = urlparse.urlsplit(fileurl).hostname is not None
    return fileurl if is_remote else path

def build_url(user, host, base, relpath=""):
    """Build remote file URL from relative path.

    Arguments:
        user (str): user name.
        host (str): host name as adressed by SSH.
        base (str): "home", "scratch" or "tmp".
        relpath (str): path relative to the host file system.
    """
    prtl = FILE_TRANSFER_PROTOCOL
    netloc = user + HOST_SEP + host
    basepath = osp.join(osp.sep, base, user)

    base_url = urlparse.urlunparse((prtl, netloc, basepath, "", "", ""))
    full_url = urlparse.urljoin(base_url, relpath)
    return full_url

def url_gio2asrun(filename):
    """Convert URL from Gio to asrun conventions.

    Arguments:
        filename (str): Input URL.

    Note:
        Implementing by deconstructing then reconstructing the URL.
        This is the safest way.
    """
    # Deconstruct Gio URL
    pres = urlparse.urlparse(filename)
    user, host, relpath = pres.username, pres.hostname, pres.path

    # Reconstruct asrun URL
    return make_remote_path(user, host, relpath)

def make_remote_path(user, host, path):
    """
    Add `hostname@user:` to path.

    Arguments:
        servcfg(dict): server configuration.
        path(str): path relative to remote machine.

    Returns:
        str: `path` preceded by `hostname@user:`
    """
    user = user + HOST_SEP if user else ""
    host = host + PATH_SEP if host or user else ""
    return user + host + path

def remote_exec(user, host, command, ignore_errors=False):
    """Execute *command* on a remote server and return the output.

    Raise OSError exception in case of failure.

    Arguments:
        user (str): User name on the remote host.
        host (str): Host name to be addressed by SSH.
        command (str): Command line to remotely execute.
        ignore_errors (bool): Do not stop in case of error.

    Returns:
        str: Output of the command.
    """
    from .utilities import debug_message

    debug_message("Command executed on {0}: {1}".format(host, command))
    if os.name == "nt":
        cmd = command
    else:
        cmd = [to_str(command)]
    if not is_localhost(host):
        use_shell = False
        cmd = [str("ssh"), "-n",
               "-o", "StrictHostKeyChecking=no", "-o", "BatchMode=yes",
               str(user + "@" if user else "") + host] + cmd
        debug_message("Full command {0}".format(cmd))
    else:
        use_shell = True
    ssh = SP.Popen(cmd, shell=use_shell, stdout=SP.PIPE, stderr=SP.PIPE)
    out, err = ssh.communicate()
    if ssh.returncode != 0 and not ignore_errors:
        raise OSError(to_str(err))
    return out
