# -*- coding: utf-8 -*-

# Copyright 2016 EDF R&D
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License Version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, you may download a copy of license
# from https://www.gnu.org/licenses/gpl-3.0.

"""Public API for parametric studies."""

from __future__ import print_function, unicode_literals

import os
import os.path as osp
import pickle
import time

import numpy

from ..common import debug_message, debug_mode, no_new_attributes, remove_path
from ..datamodel import History
from ..datamodel.engine import Engine, runner_factory
from ..datamodel.engine.engine_utils import (create_asrun_profil,
                                             from_asrun_params)
from ..datamodel.parametric import INPUTS, OUTPUTS
from ..datamodel.result import StateOptions as SO


class ParametricCalculation(object):
    """Object to execute a calculation exported by AsterStudy.

    Arguments:
        casedir (str): Path to the directory containing the Case description.
            The Case is defined by a file named 'parametric.export' in this
            directory.
        names (list[str]): List of the names of the input variables.
        values (list[float]): List of the values of the input variables.
    """

    basedir = runcdir = names = inputs = outputs = verbose = None
    _rcname = None

    __setattr__ = no_new_attributes(object.__setattr__)

    def __init__(self, casedir, names, values):
        """Initialization."""
        self.basedir = casedir
        self.names = names
        self.inputs = values
        self.outputs = []
        self.verbose = 1
        self.runcdir = osp.join(self.basedir, self.run_case_name())
        os.makedirs(self.runcdir)

    def _print(self, msg):
        """Print message on stdout (if verbosity is enabled).

        Arguments:
            msg (str): Text of the message.
        """
        if self.verbose:
            print(msg)

    def set_verbosity(self, value):
        """Set verbosity level.

        Arguments:
            value (int): 0 means silent, 1 verbose.
        """
        self.verbose = value

    def run(self):
        """Execute the calculation."""
        runcase, params = self.setup()

        runner = runner_factory(Engine.Salome, case=runcase)
        runner.start(params)
        last = runcase.results()[-1]

        # waiting for the end of the calculation
        time_limit = 30
        timeout = 5 * time_limit

        t_ini = time.time()
        # refresh at least every minute, no more than every 3 seconds
        nominal = min([max([time_limit / 10., 3.]), 60.])
        ldelay = [1., 2., nominal]
        ended = False
        while not ended:
            delay = nominal if not ldelay else ldelay.pop(0)
            self._print("Waiting for {0} s...".format(delay))
            time.sleep(delay)
            state = runner.result_state(last)
            self._print("State is: {0}".format(SO.name(state)))
            ended = runner.is_finished() or (time.time() - t_ini) > timeout

        state = runner.result_state(last)
        self._print("Execution state: {0}".format(SO.name(state)))

        self.read_output_file()
        if state & SO.Success and not debug_mode():
            self.clear_files()

    def output_values(self):
        """Return output values."""
        return self.outputs

    def setup(self):
        """Set up the execution.

        Returns:
            *RunCase*, dict: The RunCase object and its execution parameters.
        """
        history = History()
        history.folder = self.basedir
        export = osp.join(self.basedir, 'parametric.export')
        prof = create_asrun_profil(export)
        case = history.import_case(export, force_text=True)
        debug_message("ParametricCalculation case: {0}".format(case))

        # use different input/output files for each run
        for stage in case.stages:
            for _, info in stage.handle2info.iteritems():
                if osp.basename(info.filename) == INPUTS:
                    info.filename = osp.join(self.runcdir, INPUTS)
                if osp.basename(info.filename) == OUTPUTS:
                    info.filename = osp.join(self.runcdir, OUTPUTS)

        # job parameters
        params = from_asrun_params(prof)
        params['no_database'] = True

        self.create_input_file()
        runcase = history.create_run_case(name=self.run_case_name())
        return runcase, params

    def create_input_file(self):
        """Create the file to pass the input values."""
        self._print("Input variables: {0}".format(self.names))
        self._print("Input values: {0}".format(self.inputs))
        filename = osp.join(self.runcdir, INPUTS)
        with open(filename, 'wb') as pick:
            pickle.dump(self.names, pick)
            pickle.dump(self.inputs, pick)

    def read_output_file(self):
        """Read the output file to extract results."""
        filename = osp.join(self.runcdir, OUTPUTS)
        if not osp.exists(filename):
            self._print("ERROR: no such file '{0}'".format(filename))
            return
        results = numpy.load(filename)
        nbline, nbvar = results.shape
        self._print("Raw output values - {0} variable(s), {1} line(s):"
                    .format(nbvar, nbline))
        self._print(str(results))
        self.outputs = results.T
        if nbline == 1:
            self.outputs = list(results[0])

    def clear_files(self):
        """Remove temporary files."""
        remove_path(self.runcdir)
        remove_path(self._run_case_mark())

    def run_case_name(self):
        """Return a new (inexistant) RunCase name.

        Returns:
            str: Unused name for a new RunCase.
        """
        if self._rcname:
            return self._rcname

        def _name(idx, mark=False):
            name = ".run_param{0:04d}".format(idx)
            if mark:
                name = "{0}.mark".format(name)
            return osp.join(self.basedir, name)

        i = 1
        while osp.exists(_name(i)) or osp.exists(_name(i, mark=True)):
            i += 1
        name = osp.basename(_name(i))
        with open(_name(i, mark=True), 'wb') as fobj:
            fobj.write('place holder')
        self._rcname = name
        return name

    def _run_case_mark(self):
        return osp.join(self.basedir, "{0}.mark".format(self.run_case_name()))
